#include <stdio.h>
#include <strings.h>
#include <stdlib.h>
#include "neo.h"

void neo_command_device_summary(neo_global *g, int argc, char **argv) {
  neo_location loc;
  neo_device *d;
  neo_list summary;
  int ret, i, j;
  
  if (argc<1) {
    printf("You must supply a location.\n");
    return;
  }

  ret=neo_location_create_from_string(&loc, argv[0]);
  if (ret) {
    printf("Could not parse location %s\n", argv[0]);
    return;
  }

  neo_location_set_deviceconsts(&loc, g);

  j=neo_location_num_hosts(&loc);
  for (i=0; i<j; i++) {
    d=neo_location_get_device_n(&loc, i);
    if (!neo_device_is_probed(d)) {
      printf("Could not contact host %s.\n", neo_device_get_name(d));
    } else {
      neo_list_create(&summary);
      ret=neo_device_command_summary(d, &loc, &summary);
      if (ret<0) {
	printf("Could not get summary for %s which is a %s\n", neo_device_get_name(d), neo_device_get_devname(d));
      } else {
	if (ret==NEO_TYPE_BOARDLIST) {
	  summary_print_boards((neo_boardlist *) &summary);
	} else if (ret==NEO_TYPE_PORTLIST) {
	  summary_print_ports((neo_portlist *) &summary);
	} else {
	  printf("Internal error generating summary for this device.\n");
	}
      }
      if (ret==NEO_TYPE_BOARDLIST) {
	neo_boardlist_free_all(&summary);
      } else if (ret==NEO_TYPE_PORTLIST) {
	neo_portlist_free_all(&summary);
      } else {
	neo_list_free_simple(&summary);
      }
    }
  }
  neo_location_free_elements(&loc);
}

void summary_print_boards(neo_boardlist *s) {
  int i, j;
  neo_board *b;

  printf("Board summary:\n");
  printf(" b type               sg   ul ultype     \n");  
  printf("-----------------------------------------\n");
  j=neo_boardlist_get_size(s);
  for (i=0; i<j; i++) {
    b=neo_boardlist_get_element(s, i);
    printf("%2i %-17.17s ", neo_board_get_number(b), neo_board_get_descr(b));

    if (!neo_board_is_present(b)) {
      printf("\n");
      continue;
    }
    
    printf("  %1i ", neo_board_get_segment(b));

    if (neo_board_link_is_up(b)) {
      printf("%4i ", neo_board_get_speed(b)/1000000);
    } else if (neo_board_link_is_down(b)) {
      printf("     ");
    } else if (neo_board_link_is_unknown(b)){
      printf(" ??? ");
    }

    printf("%-10.10s ", neo_board_get_uplinkdescr(b));

    printf("\n");
  }

}

void summary_print_ports(neo_portlist *s) {
  int i, j;
  neo_port *p;
  printf("Port summary:\n");
  printf(" p type          u  lnk adm ap\n");  
  printf("------------------------------\n");

  j=neo_portlist_get_size(s);
  for (i=0; i<j; i++) {
    p=neo_portlist_get_element(s, i);
    printf("%2i %-13.13s ", neo_port_get_number(p), neo_port_get_descr(p));
    if (neo_port_get_uplink(p)) {
      printf("* ");
    } else {
      printf("  ");
    }
    if (neo_port_link_is_up(p) || neo_port_link_is_autopart(p)) {
      printf("%4i ", neo_port_get_speed(p)/1000000);
    } else if (neo_port_link_is_down(p)) {
      printf("   - ");
    } else {
      printf("???? ");
    }
    if (neo_port_get_admin(p) == 1) {
      printf(" On ");
    } else if (neo_port_get_admin(p) == 0) {
      printf("Off ");
    } else {
      printf("??? ");
    }

    if (neo_port_link_is_autopart(p)) {
      printf("!! ");
    }
    printf("\n");
  }
}
