#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include "neo.h"

void command_stats(int argc, char **argv) {
  int ret, j;
  neo_location loc;
  neo_device *d;
  neo_list list;
    
  argc--;
  argv++;

  if (argc<1) {
    printf("Not enough arguments for stats.\n");
    return;
  }

  ret=neo_location_create_from_string(&loc, argv[0]);
  if (ret) {
    printf("Error parsing location\n");
    return;
  }

  printf("Probing devices ...\n");
  neo_location_set_deviceconsts(&loc, g);

  j=neo_location_num_hosts(&loc);
  if (j==1) {
    /* one device */
    d=neo_location_get_first_device(&loc);

    if (!neo_device_is_probed(d)) {
      return;
    }

    if (!neo_device_has_boards(d) && neo_location_has_board(&loc)) {
      printf("%s is a %s and has no boards\n", neo_device_get_name(d), neo_device_get_devname(d));
      neo_location_free_elements(&loc);
      return;
    }

    neo_list_create(&list);
    ret=neo_device_command_stats(d, &loc, &list); /* JUST ASSUMING PORT LIST FOR THE MOMENT */
    if (ret==NEO_TYPE_PORTLIST) {
      neo_command_stats_print_portstats(&list);
      neo_portlist_free_all(&list);
    } else if (ret==NEO_TYPE_BOARDLIST) {
      neo_command_stats_print_boardstats(&list);
      neo_boardlist_free_all(&list);
    } else {
      neo_list_free_simple(&list);
    }
  } else if (j>1) {
    /* many devices */
    
  } else {
    printf("No devices found to search\n");
    neo_location_free_elements(&loc);
    return;
  }

}

void neo_command_stats_print_portstats(neo_portlist *pl) {
  int i, j;
  neo_port *p;
  neo_stats *s;

  printf("Port statistics:\n");
  printf(" p type          u  lnk adm ap   kbs   ikbs   okbs   pps  ipps  opps ierps oerps\n");  
  printf("--------------------------------------------------------------------------------\n");
  j=neo_portlist_get_size(pl);
  for (i=0; i<j; i++) {
    p=neo_portlist_get_element(pl, i);
    s=neo_port_get_stats(p);
    printf("%2i %-13.13s ", neo_port_get_number(p), neo_port_get_descr(p));
    if (neo_port_get_uplink(p)) {
      printf("* ");
    } else {
      printf("  ");
    }
    if (neo_port_link_is_up(p) || neo_port_link_is_autopart(p)) {
      printf("%4i ", neo_port_get_speed(p)/1000000);
    } else if (neo_port_link_is_down(p)) {
      printf("   - ");
    } else {
      printf(" ??? ");
    }
    if (neo_port_get_admin(p) == 1) {
      printf(" On ");
    } else if (neo_port_get_admin(p) == 0) {
      printf("Off ");
    } else {
      printf("??? ");
    }

    if (neo_port_link_is_autopart(p)) {
      printf("!! ");
    }

    if (neo_stats_get_generic(s, NEO_S_TOTALKBS, NEO_S_ERROR)) {
      printf("     ???");
    } else if (neo_stats_get_generic(s, NEO_S_TOTALKBS, NEO_S_UNAVAIL)) {
      printf("       -");
    } else {
      printf("  %6i", neo_stats_get_generic(s, NEO_S_TOTALKBS, NEO_S_INT));
    }
    
    if (neo_stats_get_generic(s, NEO_S_INKBS, NEO_S_ERROR)) {
      printf("    ???");
    } else if (neo_stats_get_generic(s, NEO_S_INKBS, NEO_S_UNAVAIL)) {
      printf("      -");
    } else {
      printf(" %6i", neo_stats_get_generic(s, NEO_S_INKBS, NEO_S_INT));
    }
    
    if (neo_stats_get_generic(s, NEO_S_OUTKBS, NEO_S_ERROR)) {
      printf("    ???");
    } else if (neo_stats_get_generic(s, NEO_S_OUTKBS, NEO_S_UNAVAIL)) {
      printf("      -");
    } else {
      printf(" %6i", neo_stats_get_generic(s, NEO_S_OUTKBS, NEO_S_INT));
    }

    if (neo_stats_get_generic(s, NEO_S_TOTALPPS, NEO_S_ERROR)) {
      printf("   ???");
    } else if (neo_stats_get_generic(s, NEO_S_TOTALPPS, NEO_S_UNAVAIL)) {
      printf("     -");
    } else {
      printf(" %5i", neo_stats_get_generic(s, NEO_S_TOTALPPS, NEO_S_INT));
    }

    if (neo_stats_get_generic(s, NEO_S_INPPS, NEO_S_ERROR)) {
      printf("   ???");
    } else if (neo_stats_get_generic(s, NEO_S_INPPS, NEO_S_UNAVAIL)) {
      printf("     -");
    } else {
      printf(" %5i", neo_stats_get_generic(s, NEO_S_INPPS, NEO_S_INT));
    }
    
    if (neo_stats_get_generic(s, NEO_S_OUTPPS, NEO_S_ERROR)) {
      printf("   ???");
    } else if (neo_stats_get_generic(s, NEO_S_OUTPPS, NEO_S_UNAVAIL)) {
      printf("     -");
    } else {
      printf(" %5i", neo_stats_get_generic(s, NEO_S_OUTPPS, NEO_S_INT));
    }

    if (neo_stats_get_generic(s, NEO_S_INERRPS, NEO_S_ERROR)) {
      printf("   ???");
    } else if (neo_stats_get_generic(s, NEO_S_INERRPS, NEO_S_UNAVAIL)) {
      printf("       -");
    } else {
      printf("  %4i", neo_stats_get_generic(s, NEO_S_INERRPS, NEO_S_INT));
    }
    if (neo_stats_get_generic(s, NEO_S_OUTERRPS, NEO_S_ERROR)) {
      printf("   ???");
    } else if (neo_stats_get_generic(s, NEO_S_OUTERRPS, NEO_S_UNAVAIL)) {
      printf("       -");
    } else {
      printf("  %4i", neo_stats_get_generic(s, NEO_S_OUTERRPS, NEO_S_INT));
    }

    printf("\n");
  }
}


void neo_command_stats_print_boardstats(neo_boardlist *bl) {
  int i, j;
  neo_board *b;
  neo_stats *s;
  
  printf("Board statistics:\n");
  printf(" b type           sg   ul ultype       kbs  ikbs  okbs   pps  ipps  opps\n");
  printf("------------------------------------------------------------------------\n");
  j=neo_boardlist_get_size(bl);
  for (i=0; i<j; i++) {
    b=neo_boardlist_get_element(bl, i);
    s=neo_board_get_stats(b);
    printf("%2i %-13.13s ", neo_board_get_number(b), neo_board_get_descr(b));

    if (!neo_board_is_present(b)) {
      printf("\n");
      continue;
    }
    
    printf("  %1i ", neo_board_get_segment(b));

    if (neo_board_link_is_up(b)) {
      printf("%4i ", neo_board_get_speed(b)/1000000);
    } else if (neo_board_link_is_down(b)) {
      printf("     ");
    } else if (neo_board_link_is_unknown(b)){
      printf("???? ");
    }

    printf("%-10.10s ", neo_board_get_uplinkdescr(b));

    if (neo_stats_get_generic(s, NEO_S_TOTALKBS, NEO_S_ERROR)) {
      printf("  ???");
    } else if (neo_stats_get_generic(s, NEO_S_TOTALKBS, NEO_S_UNAVAIL)) {
      printf("    -");
    } else {
      printf("%5i", neo_stats_get_generic(s, NEO_S_TOTALKBS, NEO_S_INT));
    }
    
    if (neo_stats_get_generic(s, NEO_S_INKBS, NEO_S_ERROR)) {
      printf("   ???");
    } else if (neo_stats_get_generic(s, NEO_S_INKBS, NEO_S_UNAVAIL)) {
      printf("     -");
    } else {
      printf(" %5i", neo_stats_get_generic(s, NEO_S_INKBS, NEO_S_INT));
    }
    
    if (neo_stats_get_generic(s, NEO_S_OUTKBS, NEO_S_ERROR)) {
      printf("   ???");
    } else if (neo_stats_get_generic(s, NEO_S_OUTKBS, NEO_S_UNAVAIL)) {
      printf("     -");
    } else {
      printf(" %5i", neo_stats_get_generic(s, NEO_S_OUTKBS, NEO_S_INT));
    }

    if (neo_stats_get_generic(s, NEO_S_TOTALPPS, NEO_S_ERROR)) {
      printf("   ???");
    } else if (neo_stats_get_generic(s, NEO_S_TOTALPPS, NEO_S_UNAVAIL)) {
      printf("     -");
    } else {
      printf(" %5i", neo_stats_get_generic(s, NEO_S_TOTALPPS, NEO_S_INT));
    }

    if (neo_stats_get_generic(s, NEO_S_INPPS, NEO_S_ERROR)) {
      printf("   ???");
    } else if (neo_stats_get_generic(s, NEO_S_INPPS, NEO_S_UNAVAIL)) {
      printf("     -");
    } else {
      printf(" %5i", neo_stats_get_generic(s, NEO_S_INPPS, NEO_S_INT));
    }

    if (neo_stats_get_generic(s, NEO_S_OUTPPS, NEO_S_ERROR)) {
      printf("   ???");
    } else if (neo_stats_get_generic(s, NEO_S_OUTPPS, NEO_S_UNAVAIL)) {
      printf("     -");
    } else {
      printf(" %5i", neo_stats_get_generic(s, NEO_S_OUTPPS, NEO_S_INT));
    }

    printf("\n");
  }

}

int neo_command_stats_standardbridge_helper(neo_device *d, neo_statstransfer *st, int ifnum) {
  char obj[1024], val[1024];
  int ret;
  unsigned long x;
  char *name;

  name=neo_device_get_name(d);
  
  /* interfaces.ifTable.ifEntry.ifInOctets.%i */
  sprintf(obj, ".1.3.6.1.2.1.2.2.1.10.%i", ifnum);
  ret=neo_snmp_request_by_variable(name, obj, val);
  if (ret && (neo_error!=NEO_ERR_SNMP_NOVAR)) {
    neo_statstransfer_set_generic(st, NEO_ST_INKBS, NEO_ST_ERROR, 0);
  } else {
    x=(atol(val)*8)/1000;
    neo_statstransfer_set_generic(st, NEO_ST_INKBS, NEO_ST_INT, x);
  }
  neo_statstransfer_set_clk_inkbs(st);

  /* interfaces.ifTable.ifEntry.ifOutOctets.%i */
  sprintf(obj, ".1.3.6.1.2.1.2.2.1.16.%i", ifnum);
  ret=neo_snmp_request_by_variable(name, obj, val);
  if (ret && (neo_error!=NEO_ERR_SNMP_NOVAR)) {
    neo_statstransfer_set_generic(st, NEO_ST_OUTKBS, NEO_ST_ERROR, 0);
  } else {
    x=(atol(val)*8)/1000;
    neo_statstransfer_set_generic(st, NEO_ST_OUTKBS, NEO_ST_INT, x);
  }
  neo_statstransfer_set_clk_outkbs(st);

  /* interfaces.ifTable.ifEntry.ifInUcastPkts.%i */
  sprintf(obj, ".1.3.6.1.2.1.2.2.1.11.%i", ifnum);
  ret=neo_snmp_request_by_variable(name, obj, val);
  if (ret && (neo_error!=NEO_ERR_SNMP_NOVAR)) {
    neo_statstransfer_set_generic(st, NEO_ST_INPPSU, NEO_ST_ERROR, 0);
  } else {
    neo_statstransfer_set_generic(st, NEO_ST_INPPSU, NEO_ST_INT, atol(val));
  }
  neo_statstransfer_set_clk_inppsu(st);

  /* interfaces.ifTable.ifEntry.ifOutUcastPkts.%i */
  sprintf(obj, ".1.3.6.1.2.1.2.2.1.17.%i", ifnum);
  ret=neo_snmp_request_by_variable(name, obj, val);
  if (ret && (neo_error!=NEO_ERR_SNMP_NOVAR)) {
    neo_statstransfer_set_generic(st, NEO_ST_OUTPPSU, NEO_ST_ERROR, 0);
  } else {
    neo_statstransfer_set_generic(st, NEO_ST_OUTPPSU, NEO_ST_INT, atol(val));
  }
  neo_statstransfer_set_clk_outppsu(st);
  
  /* interfaces.ifTable.ifEntry.ifInNUcastPkts.%i */
  sprintf(obj, ".1.3.6.1.2.1.2.2.1.12.%i", ifnum);
  ret=neo_snmp_request_by_variable(name, obj, val);
  if (ret && (neo_error!=NEO_ERR_SNMP_NOVAR)) {
    neo_statstransfer_set_generic(st, NEO_ST_INPPSNU, NEO_ST_ERROR, 0);
  } else {
    neo_statstransfer_set_generic(st, NEO_ST_INPPSNU, NEO_ST_INT, atol(val));
  }
  neo_statstransfer_set_clk_inppsnu(st);

  /* interfaces.ifTable.ifEntry.ifOutNUcastPkts.%i */
  sprintf(obj, ".1.3.6.1.2.1.2.2.1.18.%i", ifnum);
  ret=neo_snmp_request_by_variable(name, obj, val);
  if (ret && (neo_error!=NEO_ERR_SNMP_NOVAR)) {
    neo_statstransfer_set_generic(st, NEO_ST_OUTPPSNU, NEO_ST_ERROR, 0);
  } else {
    x=atol(val);
    neo_statstransfer_set_generic(st, NEO_ST_OUTPPSNU, NEO_ST_INT, atol(val));
  }
  neo_statstransfer_set_clk_outppsnu(st);

  /* interfaces.ifTable.ifEntry.ifInErrors.%i */
  sprintf(obj, ".1.3.6.1.2.1.2.2.1.14.%i", ifnum);
  ret=neo_snmp_request_by_variable(name, obj, val);
  if (ret && (neo_error!=NEO_ERR_SNMP_NOVAR)) {
    neo_statstransfer_set_generic(st, NEO_ST_INERRPS, NEO_ST_ERROR, 0);
  } else {
    neo_statstransfer_set_generic(st, NEO_ST_INERRPS, NEO_ST_INT, atol(val));
  }
  neo_statstransfer_set_clk_inerrps(st);

  /* interfaces.ifTable.ifEntry.ifOutErrors.%i */
  sprintf(obj, ".1.3.6.1.2.1.2.2.1.20.%i", ifnum);
  ret=neo_snmp_request_by_variable(name, obj, val);
  if (ret && (neo_error!=NEO_ERR_SNMP_NOVAR)) {
    neo_statstransfer_set_generic(st, NEO_ST_OUTERRPS, NEO_ST_ERROR, 0);
  } else {
    neo_statstransfer_set_generic(st, NEO_ST_OUTERRPS, NEO_ST_INT, atol(val));
  }
  neo_statstransfer_set_clk_outerrps(st);

  return(0);
}
