#include "neo.h"
#include <stdio.h>
#include <string.h>
#include <strings.h>
#include <stdlib.h>

int neo_device_create_by_name(neo_device *d, char *name) {
  strcpy(d->name, name);
  d->boardlist=NULL;
  d->portlist=NULL;
  d->constset=0;
  return(0);
}

int neo_device_set_const_from_probe(neo_device *d, neo_global *g) {
  int ret;
  char descr[1024];

  /* system.sysObjectID.0 */
  ret=neo_snmp_request_by_variable(d->name, ".1.3.6.1.2.1.1.2.0", descr);
  if (ret) {
    if (neo_error==NEO_ERR_SNMP_NOHOST) {
      printf("No such host %s.\n", neo_device_get_name(d));
      return(-1);
    } else {
      return(-1);
    }
  }
  ret=_neo_device_setconsts_from_oid(d, descr);
  if (ret) {
    return(-1);
  }

  return(0);
}

int neo_device_create_and_set_const(neo_device *d, neo_global *g, char *name) {
  neo_device_create_by_name(d, name);
  neo_device_set_const_from_probe(d, g);
  return(0);
}

int neo_device_has_boards(neo_device *d) {
  return(d->hasboards);
}

char *neo_device_get_name(neo_device *d) {
  return(d->name);
}

char *neo_device_get_devname(neo_device *d) {
  return(d->devname);
}

char *neo_device_get_fullname(neo_device *d) {
  return(d->fullname);
}

neo_boardlist *neo_device_get_boardlist(neo_device *d) {
  return(d->boardlist);
}

neo_portlist *neo_device_get_portlist(neo_device *d) {
  return(d->portlist);
}

int neo_device_is_probed(neo_device *d) {
  return(d->constset);
}

void neo_device_free_elements(neo_device *d) {
  if (d==NULL) return;
  
  if (d->boardlist) neo_boardlist_free_all(d->boardlist);
  if (d->portlist) neo_portlist_free_all(d->portlist);
}

void neo_device_free_complete(neo_device *d) {
  if (d==NULL) return;

  neo_device_free_elements(d);
  free(d);
}

void neo_device_debugprint(neo_device *d) {
  printf("Device name: %s\n", neo_device_get_name(d));
  printf("ShortType: %s\n", neo_device_get_devname(d));
  printf("LongType: %s\n", neo_device_get_fullname(d));
  printf("HasBoards: %i\n", neo_device_has_boards(d));
}

int neo_device_command_port_sed(neo_device *d, neo_location *loc, int command) {
  int ret;

  if (!d->constset) {
    neo_error=NEO_ERR_CONTACT;
    return(-1);
  }
  ret=(* d->func_port_sed)(loc, command);
  return(ret);
}

int neo_device_command_port_search(neo_device *d, neo_location *loc) {
  int ret;

  if (!d->constset) {
    neo_error=NEO_ERR_CONTACT;
    return(-1);
  }
  ret=(* d->func_port_search)(loc);
  return(ret);
}

int neo_device_command_locate(neo_device *d, neo_searchable *s, neo_locationlist *l, int uplink) {
  /* see docs for the contract. */

  if (!d->func_locate) {
    neo_error=NEO_ERR_CONTACT;
    return(-1);
  }
  return((* d->func_locate)(d, s, l, uplink));
}

int neo_device_command_summary(neo_device *d, neo_location *loc, neo_list *ret) {
  if (!d->func_summary) {
    neo_error=NEO_ERR_CONTACT;
    return(-1);
  }

  return((* d->func_summary)(d, loc, ret));
}

int neo_device_command_reset(neo_device *d) {
  return((* d->func_reset)(d));
}

int neo_device_command_load(neo_device *d, neo_location *loc, int type, int direction, char *host, char *file) {
  if (!d->func_load) {
    neo_error=NEO_ERR_CONTACT;
    return(0);
  }

  return((* d->func_load)(loc, type, direction, host, file));
}

int neo_device_command_stats(neo_device *d, neo_location *loc, neo_list *list) {
  if (!d->func_stats) {
    neo_error=NEO_ERR_CONTACT;
    return(-1);
  }
  return((* d->func_stats)(loc, list));
}

int neo_device_command_info(neo_device *d, int type) {
  if (!d->func_info) {
    neo_error=NEO_ERR_CONTACT;
    return(-1);
  }
  (* d->func_info)(d, type);
  return(0);
}

int _neo_device_setconsts_from_oid(neo_device *d, char *oid) {
  int type;

  enum {
    HUB_ASN_2072,
    HUB_ASN_1012,
    HUB_ASN_1012_BDG,
    HUB_ASN_2024,
    HUB_ASN_5324,
    HUB_ASN_6216,
    
    HUB_CTN_MRXI,
    HUB_CTN_2200,

    HUB_EXT_SUM1I,
    HUB_EXT_SUM24,
    HUB_EXT_SUM48,
    HUB_EXT_ALP,
    HUB_EXT_BD,

    HUB_HP_2524,

    HUB_CIS_C2948,
    HUB_CIS_C2950,
    HUB_CIS_C4000,
    HUB_CIS_C5000,
    HUB_CIS_C5500,

    RTR_CIS_2500_IOS,
    RTR_CIS_2600_IOS,
    RTR_CIS_3500_IOS,
    RTR_CIS_4500_IOS,
    RTR_CIS_7500_IOS,
    RTR_CIS_6509_IOS,
    RTR_CIS_8500_IOS,
    
    HUB_BST_450F,

    HUB_LUC_AP,
    HUB_LUC_AP1000,
    HUB_LUC_AP2000,
    HUB_CIS_AP,

    UPS_APC,
    UPS_MGE_COMET,

    PRINTER_HP_GENERIC,

    GENERIC_DEVICE
  };

  
  type=GENERIC_DEVICE;
  if (!oid) {
    /* set an error */
    return(-1);
  }
  
  if (!strcmp(oid, ".1.3.6.1.4.1.298.2.2.3")) {
    type=HUB_ASN_2072;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.298.2.2.1")) {
    type=HUB_ASN_1012;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.298.2.2.2")) {
    type=HUB_ASN_1012_BDG;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.298.2.2.7")) {
    type=HUB_ASN_2024;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.298.2.2.15")) {
    type=HUB_ASN_5324;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.298.2.2.16")) {
    type=HUB_ASN_6216;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.52")) {
    type=HUB_CTN_MRXI;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.52.3.9.3.4.84")) {
    type=HUB_CTN_2200;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.52.3.9.3.4.80")) {
    type=HUB_CTN_2200;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.1916.2.19")) {
    type=HUB_EXT_SUM1I;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.1916.2.7")) {
    type=HUB_EXT_SUM24;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.1916.2.6")) {
    type=HUB_EXT_SUM48;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.1916.2.17")) {
    type=HUB_EXT_ALP;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.1916.2.11")) {
    type=HUB_EXT_BD;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.11.2.3.7.11.19")) {
    type=HUB_HP_2524;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.5.42")) {
    type=HUB_CIS_C2948;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.1.325")) {
    type=HUB_CIS_C2950;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.5.46")) {
    type=HUB_CIS_C4000;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.1.448")) {
    type=HUB_CIS_C4000; /* Weird */
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.5.7")) {
    type=HUB_CIS_C5000;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.5.29")) {
    type=HUB_CIS_C5000; /* really a 5002 */
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.5.34")) {
    type=HUB_CIS_C5500;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.5.17")) {
    type=HUB_CIS_C5500;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.1.17")) {
    type=RTR_CIS_2500_IOS;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.1.185")) {
    type=RTR_CIS_2600_IOS;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.1.278")) {
    type=RTR_CIS_3500_IOS;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.1.14")) {
    type=RTR_CIS_4500_IOS;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.1.283")) {
    type=RTR_CIS_6509_IOS;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.1.46")) {
    type=RTR_CIS_7500_IOS;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.9.1.203")) {
    type=RTR_CIS_8500_IOS;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.1751.1.4.1")) {
    type=HUB_LUC_AP1000;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.762.2")) {
    type=HUB_LUC_AP1000; /* really a wavepoint II */
  } else if (!strcmp(oid, ".1.3.6.1.4.1.1751.1.4.6")) {
    type=HUB_LUC_AP2000;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.551.2.1.77")) {
    type=HUB_CIS_AP;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.318.1.3.5.4")) {
    type=UPS_APC;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.318.1.3.3.2")) {
    type=UPS_APC;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.318.1.3.2.11")) {
    type=UPS_APC;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.318.1.3.2.10")) {
    type=UPS_APC;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.705.1.4")) {
    type=UPS_MGE_COMET;
  } else if (!strcmp(oid, ".1.3.6.1.4.1.11.2.3.9.1")) {
    type=PRINTER_HP_GENERIC;
  }
    

  d->type=type;
  d->constset=1;
  
  if (type==HUB_ASN_2072) {
    strcpy(d->devname, "2072");
    strcpy(d->fullname, "Asante 2072 Repeater");
    d->hasboards=1;
    d->func_port_sed=&neo_command_port_sed_2072;
    d->func_port_search=&neo_command_port_search_2072;
    d->func_locate=&neo_command_locate_2072;
    d->func_stats=&neo_command_stats_2072;
    d->func_summary=&neo_command_device_summary_2072;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_2072;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_ASN_1012) {
    strcpy(d->devname, "1012");
    strcpy(d->fullname, "Asante 1012 Repeater");
    d->hasboards=1;
    d->func_port_sed=&neo_command_port_sed_1012;
    d->func_port_search=&neo_command_port_search_2072;
    d->func_locate=&neo_command_locate_2072;
    d->func_stats=&neo_command_stats_1012;
    d->func_summary=&neo_command_device_summary_1012;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_2072; /* TEST THIS */
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_ASN_1012_BDG) {
    strcpy(d->devname, "1012B");
    strcpy(d->fullname, "Asante 1012 Bridge");
    d->hasboards=1;
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_null;
    d->func_locate=&neo_command_locate_null;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_ASN_2024) {
    strcpy(d->devname, "NetStacker");
    strcpy(d->fullname, "Asante NetStacker Repeater");
    d->hasboards=1;   /* ? */
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_null;
    d->func_locate=&neo_command_locate_null;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_ASN_5324) {
    strcpy(d->devname, "5324");
    if (type==HUB_ASN_5324) {
      strcpy(d->fullname, "Asante 5324 IntraSwitch");
    } else {
      strcpy(d->fullname, "Asante 5324 IntraSwitch with HACK");
    }
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_5324;
    d->func_port_search=&neo_command_port_search_5324;
    d->func_locate=&neo_command_locate_5324;
    d->func_stats=&neo_command_stats_5324;
    d->func_summary=&neo_command_device_summary_5324;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_ASN_6216) {
    /**** WARNING: THIS IS NOT RIGHT, IT'S JUST A HACK FOR NOW ****/
    /**** THIS SWTICH HAS 15 ports ****/
    strcpy(d->devname, "6216");
    strcpy(d->fullname, "Asante 6216 IntraSwitch");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_null;
    d->func_locate=&neo_command_locate_null;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_CTN_2200) {
    strcpy(d->devname, "C2200");
    strcpy(d->fullname, "Cabletron 2200 Switch");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_c2200;
    d->func_port_search=&neo_command_port_search_c2200;
    d->func_locate=&neo_command_locate_c2200;
    d->func_stats=&neo_command_stats_c2200;
    d->func_summary=&neo_command_device_summary_c2200;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_CTN_MRXI) {
    strcpy(d->devname, "MRXI");
    strcpy(d->fullname, "Cabletron MRXI");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_null;
    d->func_locate=&neo_command_locate_null;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_CIS_C2948) {
    strcpy(d->devname, "Cat2948");
    strcpy(d->fullname, "Catalyst 2948 Series Switch");
    d->hasboards=1;
    d->func_port_sed=&neo_command_port_sed_cat2948;
    d->func_port_search=&neo_command_port_search_cat5000;
    d->func_locate=&neo_command_locate_cat2948;
    d->func_stats=&neo_command_stats_cat5000; /* TEST MORE THOROUGHLY */
    d->func_summary=&neo_command_device_summary_cat4000; /** THIS IS UNTESTED **/
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_cat5000;
  } else if (type==HUB_CIS_C2950) {
    strcpy(d->devname, "Cat2950");
    strcpy(d->fullname, "Catalyst 2950 Series Switch");
    d->hasboards=1;
    d->func_port_sed=&neo_command_port_sed_cat2948;
    d->func_port_search=&neo_command_port_search_cat5000;
    d->func_locate=&neo_command_locate_cat2948;
    d->func_stats=&neo_command_stats_cat5000; /* TEST MORE THOROUGHLY */
    d->func_summary=&neo_command_device_summary_generic; /** THIS IS UNTESTED **/
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_cat5000;
  } else if (type==HUB_CIS_C4000) {
    strcpy(d->devname, "Cat4000");
    strcpy(d->fullname, "Catalyst 4000 Series Switch");
    d->hasboards=1;
    d->func_port_sed=&neo_command_port_sed_cat4000; /* TEST MORE THOROUGHLY */
    d->func_port_search=&neo_command_port_search_cat5000;
    d->func_locate=&neo_command_locate_cat2948;
    d->func_stats=&neo_command_stats_cat5000; /* UNTESTED */
    d->func_summary=&neo_command_device_summary_cat4000;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_cat5000;
  } else if (type==HUB_CIS_C5000) {
    strcpy(d->devname, "Cat5000");
    strcpy(d->fullname, "Catalyst 5000 Series Switch");
    d->hasboards=1;
    d->func_port_sed=&neo_command_port_sed_cat5000;
    d->func_port_search=&neo_command_port_search_cat5000;
    d->func_locate=&neo_command_locate_cat5000;
    d->func_stats=&neo_command_stats_cat5000;
    d->func_summary=&neo_command_device_summary_cat5000;
    d->func_load=&neo_command_device_load_cat5000;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_cat5000;
  } else if (type==HUB_CIS_C5500) {
    strcpy(d->devname, "Cat5500");
    strcpy(d->fullname, "Catalyst 5500 Series Switch");
    d->hasboards=1;
    d->func_port_sed=&neo_command_port_sed_cat5000;
    d->func_port_search=&neo_command_port_search_cat5000;
    d->func_locate=&neo_command_locate_cat5000;
    d->func_stats=&neo_command_stats_cat5000;
    d->func_summary=&neo_command_device_summary_cat5000;
    d->func_load=&neo_command_device_load_cat5000;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_cat5000;
  } else if (type==RTR_CIS_7500_IOS || type==RTR_CIS_2600_IOS ||
	     type==RTR_CIS_3500_IOS || type==RTR_CIS_2500_IOS ||
	     type==RTR_CIS_4500_IOS) {
    strcpy(d->devname, "router");
    strcpy(d->fullname, "Cisco Router");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_router;
    d->func_port_search=&neo_command_port_search_router;
    d->func_locate=&neo_command_locate_router;
    d->func_stats=&neo_command_stats_router;
    d->func_summary=&neo_command_device_summary_router;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_router_old;
  } else if (type==RTR_CIS_8500_IOS || type==RTR_CIS_6509_IOS) {
    strcpy(d->devname, "router");
    strcpy(d->fullname, "Cisco Router");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_router;
    d->func_port_search=&neo_command_port_search_router;
    d->func_locate=&neo_command_locate_router;
    d->func_stats=&neo_command_stats_router;
    d->func_summary=&neo_command_device_summary_router;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_router_new;
  } else if (type==HUB_BST_450F) { /* test */
    strcpy(d->devname, "BayStack 450");
    strcpy(d->fullname, "BayStack 450");
    d->hasboards=0; /* ? */
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_null;
    d->func_locate=&neo_command_locate_null;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_LUC_AP) {
    strcpy(d->devname, "Lucent Wireless");
    strcpy(d->fullname, "Lucent Wireless Access Point");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_null;
    d->func_locate=&neo_command_locate_lucentwav;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_lucentwav;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_LUC_AP1000) {
    strcpy(d->devname, "Lucent Wireless 1000");
    strcpy(d->fullname, "Lucent Wireless Access Point 1000");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_lucentwav;
    d->func_locate=&neo_command_locate_lucentwav;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_lucentwav;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_LUC_AP2000) {
    strcpy(d->devname, "Lucent Wireless 2000");
    strcpy(d->fullname, "Lucent Wireless Access Point 2000");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_lucentwav;
    d->func_locate=&neo_command_locate_lucentwav;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_lucentwav;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_CIS_AP) {
    strcpy(d->devname, "Cisco Wireless");
    strcpy(d->fullname, "Cisco Aironet Acesss Point");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_ciscowav;
    d->func_locate=&neo_command_locate_ciscowav;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_null;
  } else if (type==HUB_EXT_SUM1I) {
    strcpy(d->devname, "Summit 1i");
    strcpy(d->fullname, "Extreme Summit 1i");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_extsum1;
    d->func_port_search=&neo_command_port_search_extsum1;
    d->func_locate=&neo_command_locate_extsum1;
    d->func_stats=&neo_command_stats_extsum1;
    d->func_summary=&neo_command_device_summary_extsum1;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_extalp;
  } else if (type==HUB_EXT_SUM24) {
    strcpy(d->devname, "Summit 24");
    strcpy(d->fullname, "Extreme Summit 24");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_extsum24;
    d->func_port_search=&neo_command_port_search_extsum48;
    d->func_locate=&neo_command_locate_extsum24;
    d->func_stats=&neo_command_stats_extsum24;
    d->func_summary=&neo_command_device_summary_extsum24;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_extalp;
  } else if (type==HUB_EXT_SUM48) {
    strcpy(d->devname, "Summit 48");
    strcpy(d->fullname, "Extreme Summit 48");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_extsum48;
    d->func_port_search=&neo_command_port_search_extsum48;
    d->func_locate=&neo_command_locate_extsum48;
    d->func_stats=&neo_command_stats_extsum48;
    d->func_summary=&neo_command_device_summary_extsum48;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_extalp;
  } else if (type==HUB_EXT_ALP) {
    strcpy(d->devname, "Apline");
    strcpy(d->fullname, "Extreme Alpine");
    d->hasboards=1;
    d->func_port_sed=&neo_command_port_sed_extalp;
    d->func_port_search=&neo_command_port_search_extalp;
    d->func_locate=&neo_command_locate_extalp;
    d->func_stats=&neo_command_stats_extalp;
    d->func_summary=&neo_command_device_summary_extalp;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_extalp;
  } else if (type==HUB_EXT_BD) {
    strcpy(d->devname, "Black Diamond");
    strcpy(d->fullname, "Extreme Black Diamond");
    d->hasboards=1;
    d->func_port_sed=&neo_command_port_sed_extalp;
    d->func_port_search=&neo_command_port_search_extalp;
    d->func_locate=&neo_command_locate_extalp;
    d->func_stats=&neo_command_stats_extalp;
    d->func_summary=&neo_command_device_summary_extalp;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_extalp;
  } else if (type==HUB_HP_2524) {
    strcpy(d->devname, "HP 2524");
    strcpy(d->fullname, "HP 2524");
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_hp2524;
    d->func_port_search=&neo_command_port_search_hp2524;
    d->func_locate=&neo_command_locate_hp2524;
    d->func_stats=&neo_command_stats_hp2524;
    d->func_summary=&neo_command_device_summary_hp2524;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_null;
  } else if (type==UPS_APC) {
    strcpy(d->devname, "APC UPS");
    strcpy(d->fullname, "APC UPS");
    d->hasboards=0;
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_null;
    d->func_locate=&neo_command_locate_null;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_apcups;
  } else if (type==UPS_MGE_COMET) {
    strcpy(d->devname, "MGE UPS");
    strcpy(d->fullname, "MGE COMET UPS");
    d->hasboards=0;
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_null;
    d->func_locate=&neo_command_locate_null;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_generic;
  } else if (type==PRINTER_HP_GENERIC) {
    strcpy(d->devname, "HP Printer");
    strcpy(d->fullname, "HP Printer");
    d->hasboards=0;
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_null;
    d->func_port_search=&neo_command_port_search_null;
    d->func_locate=&neo_command_locate_null;
    d->func_stats=&neo_command_stats_null;
    d->func_summary=&neo_command_device_summary_null;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_hpprinter;
    d->func_info=&neo_command_device_info_hpprinter;
  } else {
    d->type=GENERIC_DEVICE;
    strcpy(d->devname, "Unknown");
    strcpy(d->fullname, "Unknown Device");
    d->hasboards=0;
    d->hasboards=0;
    d->func_port_sed=&neo_command_port_sed_generic;
    d->func_port_search=&neo_command_port_search_generic;
    d->func_locate=&neo_command_locate_generic;
    d->func_stats=&neo_command_stats_generic;
    d->func_summary=&neo_command_device_summary_generic;
    d->func_load=&neo_command_device_load_null;
    d->func_reset=&neo_command_device_reset_null;
    d->func_info=&neo_command_device_info_null;
  }
  return(0);
}

