/* Written by James Kretchmar, MIT
 *
 * Copyright 2000 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice and this
 * permission notice appear in all copies and in supporting
 * documentation.  No representation is made about the suitability of
 * this software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <signal.h>
#include <setjmp.h>
#include "neo.h"
#ifdef HAVE_LIBREADLINE
#include <readline/readline.h>
#include <readline/history.h>
#endif

int main(int argc, char **argv) {
  int ret;

  /* create the global env. */
  g=malloc(sizeof(neo_global));
  neo_global_create(g);

  if (argc>0) {
    neo_global_set_argv0(g, argv[0]);
  }

  if (argc < 2) {
    interactive();
    exit(0);
  }

  /* pull out flags first */
  /* we want to replace this with getopt later */
  while (1) {
    if (!strcmp(argv[1], "-b")) {
      if (!argv[2]) {
	printf("Need an argument to -b\n");
	exit(-1);
      }
      neo_global_set_burst(g, atoi(argv[2]));
      argv+=2; argc-=2;
    } else if (!strcmp(argv[1], "-c")) {
      if (!argv[2]) {
	printf("Need an argument to -c\n");
	exit(-1);
      }
      neo_global_set_readcom(g, argv[2]);
      neo_global_set_writecom(g, argv[2]);
      argv+=2; argc-=2;
    } else if (!strcmp(argv[1], "-w")) {
      if (!argv[2]) {
	printf("Need an argument to -w\n");
	exit(-1);
      }
      neo_global_set_writecom(g, argv[2]);
      argv+=2; argc-=2;
    } else if (!strcmp(argv[1], "-r")) {
      if (!argv[2]) {
	printf("Need an argument to -r\n");
	exit(-1);
      }
      neo_global_set_readcom(g, argv[2]);
      argv+=2; argc-=2;
    } else if (!strcmp(argv[1], "-k")) {
      if (!argv[2]) {
	printf("Need an argument to -k\n");
	exit(-1);
      }
      neo_global_set_keyfile(g, argv[2]);
      argv+=2; argc-=2;
    } else if (!strcmp(argv[1], "-y")) {
      neo_global_set_areyousure_yes(g);
      argv++; argc--;
    } else if (!strcmp(argv[1], "-n")) {
      neo_global_set_areyousure_no(g);
      argv++; argc--;
    } else {
      break;
    }
  }

  ret=dispatch(g, argc-1, argv+1);
  if (ret) {
    printf("Invalid command: '%s'\n", argv[1]);
  }
  return(0);
}

void interactive() {
  int ret, argc, i;
  char *inptr;
  char **argv;
  struct sigaction sigact;
#ifndef HAVE_LIBREADLINE
  char buff[1024];
#endif
  
  /* set up the signal catcher */
  signal(SIGINT, neo_sig_handler);
  sigact.sa_handler=neo_sig_handler;
  sigemptyset(&sigact.sa_mask);
  sigact.sa_flags=0;
  sigaction(SIGINT, &sigact, NULL);

#ifdef MIT
  neo_register("startup");
#endif

  printf("\nWelcome to neo version %s.\n", neo_global_get_version(g));
#ifdef MIT  
  printf("This version built with MIT extensions.\n");
#endif
  printf("Consult 'help version' for information on features in this version.\n\n");
#ifndef HAVE_LIBREADLINE
  printf("Warning: command line editing is not available because you have.\n");
  printf("         not built with libreadline.\n\n");
#endif
  
  sigsetjmp(jb, 255);
#ifdef HAVE_LIBREADLINE
  while (1) {
    inptr=readline("neo: ");
    if (!inptr) {
      printf("\n");
      break; /* exit program */
    }
    if (*inptr) add_history(inptr);
    if (!strcmp(inptr, "")) {
      continue;
    } else if (!strcmp(inptr, "quit") || !strcmp(inptr, "exit") || !strcmp(inptr, "q")) {
      break;
    } else {
      argv=atokenize(inptr, " \t\n\r", &argc);
      if (argc<0) continue;
      ret=dispatch(g, argc, argv);
      if (ret) printf("Invalid command: '%s'\n", argv[0]);
      for (i=0; i<argc; i++) {
	free(argv[i]);
      }
      free(argv);
      free(inptr);
    }
  }
#else
  inptr=buff;
  while (1) {
    strcpy(buff, "");
    printf("neo: ");
    if (!fgets(buff, 1024, stdin)) {
      printf("\n");
      exit(0);
    }
    buff[strlen(buff)-1]='\0';
    if (!strcmp(inptr, "")) continue;
    if (!strcmp(inptr, "quit") || !strcmp(inptr, "exit") || !strcmp(inptr, "q")) break;
    argv=atokenize(inptr, " \t\n\r", &argc);
    if (argc<0) continue;
    ret=dispatch(g, argc, argv);
    if (ret) printf("Invalid command: '%s'\n", argv[0]);
  }
#endif
}

int dispatch(neo_global *g, int argc, char **argv) {
  if (argc < 1) {
    return(0);
  }

  if (commatch(argv[0], "locate", "loc", NULL)) {
    command_locate(argc, argv);
  } else if (commatch(argv[0], "stats", "stat", NULL)) {
    command_stats(argc, argv);
  } else if (commatch(argv[0], "port", NULL)) {
    command_port(argc, argv);
  } else if (commatch(argv[0], "board", NULL)) {
    command_board(argc, argv);
  } else if (commatch(argv[0], "arpfind", "arpf", "arp", NULL)) {
    command_arpfind(argc, argv);
  } else if (commatch(argv[0], "device", "dev", NULL)) {
    command_device(argc, argv);
  } else if (commatch(argv[0], "location", NULL)) {
    command_location(argc, argv);
  } else if (commatch(argv[0], "print", "pr", NULL)) {
    command_print(argc, argv);
  } else if (commatch(argv[0], "set", NULL)) {
    command_set(argc, argv);
  } else if (commatch(argv[0], "stats", NULL)) {
    command_stats(argc, argv);
  } else if (commatch(argv[0], "help", "?", NULL)) {
    command_help(argc, argv);
  } else if (commatch(argv[0], "exec", NULL)) {
    command_exec(argc, argv);
  } else if (commatch(argv[0], "hostinfo", "host", NULL)) {
    command_hostinfo(argc, argv);
  } else if (commatch(argv[0], "ktest", NULL)) {
    command_ktest(argc, argv);
  } else {
    return(-1);
  }
  return(0);
}


void command_ktest(int argc, char **argv) {

}


void command_winky(int argc, char **argv) {
  printf("\n\n\n                    Bobby!\n\n\n\n");
}


void neo_error_print(int n) {
  char *neo_error_string[] = {
    "No error",
    "Generic error",
    "Bad format",
    "No such host",
    "Connection timed out with host",
    "Could not contact host",
    
    "No such SNMP variable",
    "Generic SNMP error"
  };

  printf("%s\n", neo_error_string[n]);
}

void neo_sig_handler(int sig) {
  if (sig==SIGINT) {
    printf("Interrupted!\n");
    siglongjmp(jb, 1);
  }
}
