#include <sys/types.h>
#include <time.h>
#include <regex.h>

#define LINE 1024
#define DEBUG 0
#define REPLACESTR "___"
#define VERSION "1.0.8"

typedef struct _oak_list {
  void *element;
  struct _oak_list *next;
} oak_list;

struct queue {
  char *name;
  char **msgs;
  char **host;
  int *msgcount;
  int nummsgs;
  struct fire *fire;
  oak_list *actionlist;
  int locktime;
  char *header;
  int trashqueue;
  int prescan;
};

struct action {
  char *name;
  int argc;
  char **argv;
  int haslimits;
  int limit_lines;
  int limit_linelen;
  int limit_numhosts;
  int limit_hostents;
};

struct fire {
  time_t next;       /* the unix time at which to fire next */
  time_t interval;   /* the number of seconds between intervals */
};

struct queuelist {
  struct queue *queue;
  struct queuelist *next;
};

struct matchline {
  regex_t regex;
  struct queuelist *ql;
  struct queuelist *locked_queuelist;
  struct timelist  *lock_times;
};

struct matchlist {
  struct matchline *line;
  struct matchlist *next;
};

struct timelist {
  time_t *time;
  struct timelist *next;
};

struct global {
  char *logfile;
  struct queuelist *queuelist;
  struct matchlist *matchlist;
  int nukepid;
  int nukeciscopid;
};

/* -------------------------------- action.c -------------------------------- */
struct action *action_create(char *name, int argc, char **argv);
struct action *action_create_from_argstring(char *name, char *args);
void action_set_limits(struct action *act, int lines, int linelen, int numhosts, int hostents);
void action_execute(struct action *act, struct queue *q);
char *action_get_name(struct action *act);

/* -------------------------------- fire.c -------------------------------- */
struct fire *fire_create(char *in);
int fire_is_time(struct fire *f);

/* -------------------------------- list.c -------------------------------- */
oak_list *oak_list_create();
void oak_list_append_element(oak_list *list, void *in);
void *oak_list_get_element(oak_list *list, int index);
int oak_list_get_size(oak_list *list);

/* -------------------------------- matchline.c -------------------------------- */
struct matchline *matchline_create(struct global *g, char *regex, char *queuenames);
void matchline_set_queuelist(struct matchline *m, struct queuelist *ql);
int matchline_set_queuelist_from_string(struct global *g, struct matchline *m, char *queuenames);
int matchline_string_matches(struct matchline *m, char *instr, char *out, int nukepid, int nukeciscopid);
struct queuelist *matchline_get_queuelist(struct matchline *m);
int matchline_locked_for_queue(struct matchline *m, struct queue *q);
void matchline_set_lock(struct matchline *m, struct queue *q);

/* -------------------------------- matchlist.c -------------------------------- */
struct matchlist *matchlist_append_matchline(struct matchlist *a, struct matchline *ml);
int matchlist_get_nummatches(struct matchlist *m);
struct matchline *matchlist_get_matchline_n(struct matchlist *m, int n);

/* -------------------------------- oak.c -------------------------------- */
int main(int argc, char **argv);
struct global *readconfig(char *file);
char *get_line_from_log(char *log);
char *get_host_from_log(char *log);
void print_usage(int err);

/* -------------------------------- queue.c -------------------------------- */
struct queue *queue_create(char *name, oak_list *actionlist, struct fire *fire);
char *queue_get_name(struct queue *q);
int queue_is_prescan(struct queue *q);
void queue_set_prescan(struct queue *q);
int queue_get_nummsgs(struct queue *q);
char *queue_get_msg_n(struct queue *q, int n);
char *queue_get_host_n(struct queue *q, int n);
int queue_get_msgcount_n(struct queue *q, int n);
void queue_set_header(struct queue *q, char *header);
char *queue_get_header(struct queue *q);
int queue_is_locking(struct queue *q);
void queue_set_locktime_fromstring(struct queue *q, char *in);
int queue_get_locktime(struct queue *q);
void queue_add_msg(struct queue *q, char *msg, char *host);
void queue_delete_msgs(struct queue *q);
struct fire *queue_get_fire(struct queue *q);
oak_list *queue_get_actionlist(struct queue *q);
void queue_set_fire(struct queue *q, struct fire *fire);
void queue_set_actionlist(struct queue *q, oak_list *actionlist);
char *queue_to_string(struct queue *q);
char *queue_to_string_truncate(struct queue *q, int lines, int linelen, int numhosts, int hostents);

/* -------------------------------- queuelist.c -------------------------------- */
struct queuelist *queuelist_append_queue(struct queuelist *a, struct queue *q);
int queuelist_get_num_queues(struct queuelist *ql);
struct queue *queuelist_get_queue_n(struct queuelist *ql, int n);
struct queue *queuelist_get_queue_by_name(struct queuelist *ql, char *name);
struct queuelist *queuelist_remove_queue_n(struct queuelist *ql, int n);

/* -------------------------------- timelist.c -------------------------------- */
struct timelist *timelist_append_time(struct timelist *a, time_t *t);
int timelist_get_num_times(struct timelist *tl);
time_t *timelist_get_time_n(struct timelist *tl, int n);
struct timelist *timelist_remove_time_n(struct timelist *tl, int n);

/* -------------------------------- util.c -------------------------------- */
int trim_string_by_lines(char *instr, char *outstr, int lines);
void replace_string_sections(char *in, regmatch_t pmatch[], char *out, char *newstr);
char **argstring_to_argv(char *in, int *outargc);

/* -------------------------------- zephyr.c -------------------------------- */
void send_zephyr(char *class, char *instance, char *recipient, char *message, char *opcode, char *zsig);
