#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <com_err.h>
#include <time.h>
#include "owl.h"

void owl_function_adminmsg(char *header, char *body) {
  owl_message *m;
  
  m=malloc(sizeof(owl_message));
  owl_message_create_admin(m, header, body);
  owl_messagelist_append_element(owl_global_get_msglist(&g), m);

  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
  if (owl_popwin_is_active(owl_global_get_popwin(&g))) {
    owl_popwin_refresh(owl_global_get_popwin(&g));
  }
  
  wnoutrefresh(owl_global_get_curs_recwin(&g));
  owl_global_set_needrefresh(&g);
}


void owl_function_nextmsg() {
  int curmsg;
  curmsg=owl_global_get_curmsg(&g);
  curmsg++;
  if (curmsg>owl_messagelist_get_size(owl_global_get_msglist(&g))-1) {
    curmsg=owl_messagelist_get_size(owl_global_get_msglist(&g))-1;
    if (curmsg<0) curmsg=0;
    owl_function_beep();
    owl_function_makemsg("already at last message");
  }
  owl_global_set_curmsg(&g, curmsg);
  owl_function_calculate_topmsg(OWL_DIRECTION_DOWNWARDS);

  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
  owl_global_set_direction_downwards(&g);
}


void owl_function_prevmsg() {
  int curmsg;
  curmsg=owl_global_get_curmsg(&g);
  curmsg--;
  if (curmsg<0) {
    curmsg=0;
    owl_function_beep();
    owl_function_makemsg("already at first message");
  }
  owl_global_set_curmsg(&g, curmsg);
  owl_function_calculate_topmsg(OWL_DIRECTION_UPWARDS);
  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
  owl_global_set_direction_upwards(&g);
}


void owl_function_nextmsg_notdeleted() {
  int curmsg;
  owl_message *m;

  curmsg=owl_global_get_curmsg(&g);
  while (1) {
    curmsg++;

    /* if we're out of bounds get in bounds and stop */
    if (curmsg>owl_messagelist_get_size(owl_global_get_msglist(&g))-1) {
      curmsg=owl_messagelist_get_size(owl_global_get_msglist(&g))-1;
      if (curmsg<0) curmsg=0;
      owl_function_makemsg("already at last non-deleted message");
      break;
    }

    /* if this one is not deleted we can stop where we are */
    m=owl_messagelist_get_element(owl_global_get_msglist(&g), curmsg);
    if (!owl_message_is_delete(m)) {
      break;
    }
  }
  
  owl_global_set_curmsg(&g, curmsg);
  owl_function_calculate_topmsg(OWL_DIRECTION_DOWNWARDS);
  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
  owl_global_set_direction_downwards(&g);
}


void owl_function_prevmsg_notdeleted() {
  int curmsg;
  owl_message *m;

  curmsg=owl_global_get_curmsg(&g);
  while(1) {
    curmsg--;

    /* if we're out of bounds get in bounds and stop */
    if (curmsg<0) {
      curmsg=0;
      owl_function_makemsg("already at first non-deleted message");
      break;
    }

    /* if this one is not deleted we can stop where we are */
    m=owl_messagelist_get_element(owl_global_get_msglist(&g), curmsg);
    if (!owl_message_is_delete(m)) {
      break;
    }
  }

  owl_global_set_curmsg(&g, curmsg);
  owl_function_calculate_topmsg(OWL_DIRECTION_UPWARDS);
  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
  owl_global_set_direction_upwards(&g);
}


void owl_function_deletecur() {
  int curmsg;

  /* bail if there's no current message */
  if (owl_messagelist_get_size(owl_global_get_msglist(&g)) < 1) {
    owl_function_makemsg("No current message to delete");
    return;
  }

  /* mark the message for deletion */
  curmsg=owl_global_get_curmsg(&g);
  owl_messagelist_delete_element(owl_global_get_msglist(&g), curmsg);

  /* move the poiner in the appropriate direction to the next undeleted msg */
  if (owl_global_get_direction(&g)==OWL_DIRECTION_UPWARDS) {
    owl_command_prev_notdeleted();
  } else {
    owl_command_next_notdeleted();
  }
}


void owl_function_undeletecur() {
  int curmsg;
  
  if (owl_messagelist_get_size(owl_global_get_msglist(&g)) < 1) {
    owl_function_makemsg("No current message to undelete");
    return;
  }
  curmsg=owl_global_get_curmsg(&g);

  owl_messagelist_undelete_element(owl_global_get_msglist(&g), curmsg);

  if (owl_global_get_direction(&g)==OWL_DIRECTION_UPWARDS) {
    if (curmsg>0) {
      owl_command_prev();
    } else {
      owl_command_next();
    }
  } else {
    owl_command_next();
  }

  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
}


void owl_function_expunge() {
  int curmsg;

  owl_messagelist_expunge(owl_global_get_msglist(&g));

  curmsg=owl_global_get_curmsg(&g);
  if (curmsg>owl_messagelist_get_size(owl_global_get_msglist(&g))-1) {
    owl_global_set_curmsg(&g, owl_messagelist_get_size(owl_global_get_msglist(&g))-1);
    if (owl_global_get_curmsg(&g)<0) {
      owl_global_set_curmsg(&g, 0);
    }
    owl_function_calculate_topmsg(OWL_DIRECTION_NONE);
  }

  /* if there are no messages set the direction to down in case we
     delete everything upwards */
  owl_global_set_direction_downwards(&g);
  
  owl_function_makemsg("Messages expunged");
  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
}


void owl_function_firstmsg() {
  owl_global_set_curmsg(&g, 0);
  owl_global_set_topmsg(&g, 0);
  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
  owl_global_set_direction_downwards(&g);
}


void owl_function_lastmsg() {
  int curmsg;
  
  curmsg=owl_messagelist_get_size(owl_global_get_msglist(&g))-1;
  if (curmsg<0) curmsg=0;
  owl_global_set_curmsg(&g, curmsg);
  owl_global_set_topmsg(&g, curmsg);
  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
  owl_global_set_direction_upwards(&g);
}


void owl_function_shift_right() {
  owl_global_set_rightshift(&g, owl_global_get_rightshift(&g)+10);
  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
  owl_global_set_needrefresh(&g);
}


void owl_function_shift_left() {
  int shift;

  shift=owl_global_get_rightshift(&g);
  if (shift>=10) {
    owl_global_set_rightshift(&g, shift-10);
    owl_mainwin_redisplay(owl_global_get_mainwin(&g));
    owl_global_set_needrefresh(&g);
  } else {
    owl_function_beep();
    owl_function_makemsg("Already full left");
  }
}


void owl_function_unsub() {
  unsuball();
  owl_function_makemsg("Unsubscribed from all messages.");
}


void owl_function_loadsubs(char *file) {
  int ret;
  ret=loadsubs(file);
  if (ret==0) {
    owl_function_makemsg("Subscribed to messages from file.");
  } else if (ret==-1) {
    owl_function_makemsg("Could not open file.");
  } else {
    owl_function_makemsg("Error subscribing to messages from file.");
  }
}


void owl_function_suspend() {
  endwin();
  printf("\n");
  kill(getpid(), SIGSTOP);

  /* resize to reinitialize all the windows when we come back */
  owl_command_resize();
}


void owl_function_zaway_on(char *msg) {
  char buff[1024];
  /* turn on zaway.  If msg!=NULL then
     use msg as the zaway message */
  if (msg) {
    owl_global_set_zaway_msg(&g, msg);
  } else {
    owl_global_set_zaway_msg_default(&g);
  }
  owl_global_set_zaway_on(&g);
  owl_global_set_needrefresh(&g);
  sprintf(buff, "zaway set (%s)", owl_global_get_zaway_msg(&g));
  owl_function_makemsg(buff);
}


void owl_function_zaway_off() {
  owl_global_set_zaway_off(&g);
  owl_function_makemsg("zaway off");
}


void owl_function_quit() {
  /* zlog out if we need to */
  if (owl_global_get_shutdownlogout(&g)) {
    owl_function_zlog_out();
  }

  /* execute the commands in shutdown */
  owl_config_execute("owl::shutdown();");

  /* final clean up */
  unsuball();
  ZClosePort();
  endwin();
  owl_function_debugmsg("Quitting Owl");
  exit(0);
}


void owl_function_zlog_in() {
  char *exposure, *eset;
  int ret;

  eset=EXPOSE_REALMVIS;
  exposure=ZGetVariable("exposure");
  if (exposure==NULL) {
    eset=EXPOSE_REALMVIS;
  } else if (!strcasecmp(exposure,EXPOSE_NONE)) {
    eset = EXPOSE_NONE;
  } else if (!strcasecmp(exposure,EXPOSE_OPSTAFF)) {
    eset = EXPOSE_OPSTAFF;
  } else if (!strcasecmp(exposure,EXPOSE_REALMVIS)) {
    eset = EXPOSE_REALMVIS;
  } else if (!strcasecmp(exposure,EXPOSE_REALMANN)) {
    eset = EXPOSE_REALMANN;
  } else if (!strcasecmp(exposure,EXPOSE_NETVIS)) {
    eset = EXPOSE_NETVIS;
  } else if (!strcasecmp(exposure,EXPOSE_NETANN)) {
    eset = EXPOSE_NETANN;
  }
   
  ret=ZSetLocation(eset);
  if (ret != ZERR_NONE) {
    /*
      char buff[LINE];
      sprintf(buff, "Error setting location: %s", error_message(ret));
      owl_function_makemsg(buff);
    */
  }
}

void owl_function_zlog_out() {
  int ret;
  
  ret=ZUnsetLocation();
  if (ret != ZERR_NONE) {
    /*
      char buff[LINE];
      sprintf(buff, "Error unsetting location: %s", error_message(ret));
      owl_function_makemsg(buff);
    */
  }
}


void owl_function_makemsg(char *in) {
  werase(owl_global_get_curs_msgwin(&g));
  waddstr(owl_global_get_curs_msgwin(&g), in);
  wnoutrefresh(owl_global_get_curs_msgwin(&g));
  owl_global_set_needrefresh(&g);
}


void owl_function_openurl() {
  /* visit the first url in the current message */
  owl_message *m;
  char *ptr1, *ptr2, *text, url[LINE], tmpbuff[LINE];

  if (owl_messagelist_get_size(owl_global_get_msglist(&g))==0) {
    owl_function_makemsg("No current message selected");
    return;
  }

  m=owl_messagelist_get_element(owl_global_get_msglist(&g),
				owl_global_get_curmsg(&g));
  text=owl_message_get_text(m);

  /* First look for a good URL */  
  if ((ptr1=strstr(text, "http://"))!=NULL) {
    ptr2=strpbrk(ptr1, " \n\t");
    if (ptr2) {
      strncpy(url, ptr1, ptr2-ptr1+1);
      url[ptr2-ptr1+1]='\0';
    } else {
      strcpy(url, ptr1);
    }

    /* if we had <http strip a trailing > */
    if (ptr1>text && ptr1[-1]=='<') {
      if (url[strlen(url)-1]=='>') {
	url[strlen(url)-1]='\0';
      }
    }
  } else if ((ptr1=strstr(text, "https://"))!=NULL) {
    /* Look for an https URL */  
    ptr2=strpbrk(ptr1, " \n\t");
    if (ptr2) {
      strncpy(url, ptr1, ptr2-ptr1+1);
      url[ptr2-ptr1+1]='\0';
    } else {
      strcpy(url, ptr1);
    }
    
    /* if we had <http strip a trailing > */
    if (ptr1>text && ptr1[-1]=='<') {
      if (url[strlen(url)-1]=='>') {
	url[strlen(url)-1]='\0';
      }
    }
  } else if ((ptr1=strstr(text, "www."))!=NULL) {
    /* if we can't find a real url look for www.something */
    ptr2=strpbrk(ptr1, " \n\t");
    if (ptr2) {
      strncpy(url, ptr1, ptr2-ptr1+1);
      url[ptr2-ptr1+1]='\0';
    } else {
      strcpy(url, ptr1);
    }
  } else {
    owl_function_beep();
    owl_function_makemsg("Could not find URL to open.");
    return;
  }
  
  /* open the page */
  sprintf(tmpbuff, "Opening %s ...", url);
  owl_function_makemsg(tmpbuff);
  sprintf(tmpbuff, "netscape -remote \"openURL(%s)\" > /dev/null 2> /dev/null", url);
  system(tmpbuff);
}

void owl_function_calculate_topmsg(int direction) {
  int recwinlines, y, savey, i, j, topmsg, curmsg, foo;
  owl_messagelist *msglist;
  owl_mainwin *mw;

  mw=owl_global_get_mainwin(&g);

  msglist=owl_global_get_msglist(&g);
  topmsg=owl_global_get_topmsg(&g);
  curmsg=owl_global_get_curmsg(&g);
  recwinlines=owl_global_get_recwin_lines(&g);

  if (owl_messagelist_get_size(owl_global_get_msglist(&g)) < 1) {
    return;
  }
  
  /* Find number of lines from top to bottom of curmsg (store in savey) */
  savey=0;
  for (i=topmsg; i<=curmsg; i++) {
    savey+=owl_message_get_numlines(owl_messagelist_get_element(msglist, i));
  }

  /* If our bottom line is less than 1/4 down the screen then scroll up */
  if (direction == OWL_DIRECTION_UPWARDS || direction == OWL_DIRECTION_NONE) {
    if (savey < (recwinlines / 4)) {
      y=0;
      for (j=curmsg; j>0; j--) {
	foo=owl_message_get_numlines(owl_messagelist_get_element(msglist, j));
	/* will we run the curmsg off the screen? */
	if ((foo+y) >= recwinlines) {
	  j++;
	  if (j>curmsg) j=curmsg;
	  break;
	}
	/* have saved 1/2 the screen space? */
	y+=foo;
	if (y > (recwinlines / 2)) break;
      }
      owl_global_set_topmsg(&g, j);
      return;
    }
  }

  /* If our bottom line is more than 3/4 down the screen then scroll down */
  if (direction == OWL_DIRECTION_DOWNWARDS || direction == OWL_DIRECTION_NONE) {
    if (savey > ((recwinlines * 3)/4)) {
      y=0;
      /* count lines until we can save 1/2 the screen size */
      for (j=topmsg; j<curmsg; j++) {
	y+=owl_message_get_numlines(owl_messagelist_get_element(msglist, j));
	if (y > (recwinlines / 2)) break;
      }
      if (j==curmsg) {
	j--;
      }
      owl_global_set_topmsg(&g, j+1);
      return;
    }
  }
}


void owl_function_resize() {
  owl_global_set_resize_pending(&g);
}


void owl_function_run_buffercommand() {
  char *buff;

  buff=owl_global_get_buffercommand(&g);
  if (!strncmp(buff, "zwrite ", 7)) {

    owl_command_zwrite(buff);
  }
}

void owl_function_debugmsg(char *in) {
  FILE *file;
  time_t now;
  char buff1[LINE], buff2[LINE];
  

  if (!owl_global_is_debug(&g)) return;

  file=fopen(OWL_DEBUG_FILE, "a");
  if (!file) return;

  now=time(NULL);
  strcpy(buff1, ctime(&now));
  buff1[strlen(buff1)-1]='\0';

  owl_global_get_runtime(&g, buff2);
  
  fprintf(file, "[%i -  %s - %s]: %s\n", (int) getpid(), buff1, buff2, in);
  fclose(file);
}


void owl_function_refresh() {
  owl_function_resize();
}

void owl_function_beep() {
  if (owl_global_is_bell(&g)) {
    beep();
  }
}


void owl_function_subscribe(char *class, char *inst, char *recip) {
  int ret;

  ret=owl_zephyr_sub(class, inst, recip);
  if (ret) {
    owl_function_makemsg("Error subscribing.");
  } else {
    owl_function_makemsg("Subscribed.");
  }
}


void owl_function_set_cursor(WINDOW *win) {
  wnoutrefresh(win);
}


void owl_function_full_redisplay() {
  redrawwin(owl_global_get_curs_recwin(&g));
  redrawwin(owl_global_get_curs_sepwin(&g));
  redrawwin(owl_global_get_curs_typwin(&g));
  redrawwin(owl_global_get_curs_msgwin(&g));

  wnoutrefresh(owl_global_get_curs_recwin(&g));
  wnoutrefresh(owl_global_get_curs_sepwin(&g));
  wnoutrefresh(owl_global_get_curs_typwin(&g));
  wnoutrefresh(owl_global_get_curs_msgwin(&g));
  
  sepbar("");
  owl_function_makemsg("");

  owl_global_set_needrefresh(&g);
}


void owl_function_popless_text(char *text) {
  owl_popwin *pw;
  owl_viewwin *v;

  pw=owl_global_get_popwin(&g);
  v=owl_global_get_viewwin(&g);

  owl_popwin_up(pw);
  owl_viewwin_init_text(v, owl_popwin_get_curswin(pw),
			owl_popwin_get_lines(pw), owl_popwin_get_cols(pw),
			text);
  owl_popwin_set_handler(pw, pophandler_viewwin);
  owl_popwin_refresh(pw);
  owl_viewwin_redisplay(v, 0);
  owl_global_set_needrefresh(&g);
}


void owl_function_popless_fmtext(owl_fmtext *fm) {
  owl_popwin *pw;
  owl_viewwin *v;

  pw=owl_global_get_popwin(&g);
  v=owl_global_get_viewwin(&g);

  owl_popwin_up(pw);
  owl_viewwin_init_fmtext(v, owl_popwin_get_curswin(pw),
		   owl_popwin_get_lines(pw), owl_popwin_get_cols(pw),
		   fm);
  owl_popwin_set_handler(pw, pophandler_viewwin);
  owl_popwin_refresh(pw);
  owl_viewwin_redisplay(v, 0);
  owl_global_set_needrefresh(&g);
}


void owl_function_help() {
  char buff[1024];

  strcpy(buff, "OWL HELP\n\n");
  strcat(buff, "  Keys:\n");
  strcat(buff, "    n             Move to next non-deleted message\n");
  strcat(buff, "    p             Move to previous non-deleted message\n");
  strcat(buff, "    C-n , down    Move to next message\n");
  strcat(buff, "    C-p , up      Move to previous message\n");
  strcat(buff, "    < , >         Move to first, last message\n");
  strcat(buff, "    right , left  Scroll screen left or right\n");
  strcat(buff, "    i             Print more information about a message\n");
  strcat(buff, "\n");
  strcat(buff, "    d             Mark message for deleted\n");
  strcat(buff, "    u             Undelete a message marked for deletion\n");
  strcat(buff, "    x             Expunge deleted messages\n");
  strcat(buff, "\n");
  strcat(buff, "    z             Start a zwrite command\n");
  strcat(buff, "    r             Reply to the current message\n");
  strcat(buff, "\n");
  strcat(buff, "    A             Toggle zaway\n");
  strcat(buff, "    w             Open a URL in the message in netscape\n");
  strcat(buff, "    C-l           Refresh the screen\n");
  strcat(buff, "    C-z           Suspend\n");
  strcat(buff, "    h             Print this help message\n");
  strcat(buff, "    :             Enter command mode\n");
  strcat(buff, "\nHelp on commands and other documentation is forthcoming\n\n");
  strcat(buff, "Type 'q' to exit.");

  owl_function_popless_text(buff);
}


void owl_function_info() {
  owl_message *m;
  ZNotice_t *n;
  char buff[2048], tmpbuff[1024];
  char *ptr;
  int i, j, fields, len;
  
  if (owl_messagelist_get_size(owl_global_get_msglist(&g))==0) {
    owl_function_popless_text("No message selected\n");
    return;
  }

  m=owl_messagelist_get_element(owl_global_get_msglist(&g), owl_global_get_curmsg(&g));
  if (!owl_message_is_zephyr(m)) {
    owl_function_popless_text("No zephyr info for this message\n\nPress 'q' to exit.");
    return;
  }

  n=owl_message_get_notice(m);

  sprintf(buff,   "Class     : %s\n", n->z_class);
  sprintf(buff, "%sInstance  : %s\n", buff, n->z_class_inst);
  sprintf(buff, "%sSender    : %s\n", buff, n->z_sender);
  sprintf(buff, "%sRecip     : %s\n", buff, n->z_recipient);
  sprintf(buff, "%sOpcode    : %s\n", buff, n->z_opcode);
  strcat(buff,    "Kind      : ");
  if (n->z_kind==UNSAFE) {
    strcat(buff, "UNSAFE\n");
  } else if (n->z_kind==UNACKED) {
    strcat(buff, "UNACKED\n");
  } else if (n->z_kind==ACKED) {
    strcat(buff, "ACKED\n");
  } else if (n->z_kind==HMACK) {
    strcat(buff, "HMACK\n");
  } else if (n->z_kind==HMCTL) {
    strcat(buff, "HMCTL\n");
  } else if (n->z_kind==SERVACK) {
    strcat(buff, "SERVACK\n");
  } else if (n->z_kind==SERVNAK) {
    strcat(buff, "SERVNAK\n");
  } else if (n->z_kind==CLIENTACK) {
    strcat(buff, "CLIENTACK\n");
  } else if (n->z_kind==STAT) {
    strcat(buff, "STAT\n");
  } else {
    strcat(buff, "ILLEGAL VALUE\n");
  }
  sprintf(buff, "%sTime      : %s\n", buff, owl_message_get_timestr(m));
  sprintf(buff, "%sHost      : %s\n", buff, owl_message_get_hostname(m));
  sprintf(buff, "%sPort      : %i\n", buff, n->z_port);
  strcat(buff,    "Auth      : ");
  if (n->z_auth == ZAUTH_FAILED) {
    strcat(buff, "FAILED\n");
  } else if (n->z_auth == ZAUTH_NO) {
    strcat(buff, "NO\n");
  } else if (n->z_auth == ZAUTH_YES) {
    strcat(buff, "YES\n");
  } else {
    sprintf(buff, "%sUnknown State (%i)\n", buff, n->z_auth);
  }
  sprintf(buff, "%sCheckd Ath: %i\n", buff, n->z_checked_auth);
  sprintf(buff, "%sMulti notc: %s\n", buff, n->z_multinotice);
  sprintf(buff, "%sNum other : %i\n", buff, n->z_num_other_fields);
  sprintf(buff, "%sMsg Len   : %i\n", buff, n->z_message_len);

  sprintf(buff, "%sFields    : %i\n", buff, owl_zephyr_get_num_fields(n));

  fields=owl_zephyr_get_num_fields(n);
  for (i=0; i<fields; i++) {
    sprintf(buff, "%sField %i   : ", buff, i+1);
    ptr=owl_zephyr_get_field(n, i+1, &len);
    if (!ptr) break;
    strncpy(tmpbuff, ptr, len);
    tmpbuff[len]='\0';

    /* just for testing for now */
    for (j=0; j<strlen(tmpbuff); j++) {
      if (tmpbuff[j]=='\n') tmpbuff[j]='~';
      if (tmpbuff[j]=='\r') tmpbuff[j]='!';
    }

    if (strlen(tmpbuff)>30) {
      strncat(buff, tmpbuff, 30);
      strcat(buff, "...");
    } else {
      strcat(buff, tmpbuff);
    }
    strcat(buff, "\n");
  }
  sprintf(buff, "%sDefault Fm: %s\n", buff, n->z_default_format);


  strcat(buff, "\n\nPress 'q' to exit.");
	
  owl_function_popless_text(buff);
}


void owl_function_curmsg_to_popwin() {
  owl_popwin *pw;
  owl_viewwin *v;
  owl_message *m;

  pw=owl_global_get_popwin(&g);
  v=owl_global_get_viewwin(&g);

  m=owl_messagelist_get_element(owl_global_get_msglist(&g), owl_global_get_curmsg(&g));
  owl_function_popless_fmtext(owl_message_get_fmtext(m));
}


void owl_function_page_curmsg(int step) {
  /* scroll down or up within the current message */

  int offset, curmsg, lines;
  owl_messagelist *mlist;
  owl_message *m;
  
  offset=owl_global_get_curmsg_vert_offset(&g);

  mlist=owl_global_get_msglist(&g);
  if (owl_messagelist_get_size(mlist)==0) return;
  curmsg=owl_global_get_curmsg(&g);
  m=owl_messagelist_get_element(mlist, curmsg);
  lines=owl_message_get_numlines(m);
  
  
  /* don't scroll past the last line */
  if (step>0) {
    if (offset+step > lines-1) {
      owl_global_set_curmsg_vert_offset(&g, lines-1);
    } else {
      owl_global_set_curmsg_vert_offset(&g, offset+step);
    }
  }

  /* would we be before the beginning of the message? */
  if (step<0) {
    if (offset+step<0) {
      owl_global_set_curmsg_vert_offset(&g, 0);
    } else {
      owl_global_set_curmsg_vert_offset(&g, offset+step);
    }
  }
  
  /* redisplay */
  owl_mainwin_redisplay(owl_global_get_mainwin(&g));
  owl_global_set_needrefresh(&g);
}

void owl_function_resize_typwin(int newsize) {
  owl_global_set_typwin_lines(&g, newsize);
  owl_function_resize();
}

void owl_function_typwin_grow() {
  int i;

  i=owl_global_get_typwin_lines(&g);
  owl_function_resize_typwin(i+1);
}

void owl_function_typwin_shrink() {
  int i;

  i=owl_global_get_typwin_lines(&g);
  if (i>2) {
    owl_function_resize_typwin(i-1);
  }
}

void owl_function_mainwin_pagedown() {
  int i;

  i=owl_mainwin_get_last_msg(owl_global_get_mainwin(&g));
  if (i<0) return;

  owl_global_set_curmsg(&g, i);
  owl_function_nextmsg();
}

void owl_function_mainwin_pageup() {
  owl_global_set_curmsg(&g, owl_global_get_topmsg(&g));
  owl_function_prevmsg();
}
