/* Written by James Kretchmar, MIT
 *
 * Copyright 2001 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice and this
 * permission notice appear in all copies and in supporting
 * documentation.  No representation is made about the suitability of
 * this software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <zephyr/zephyr.h>
#include <com_err.h>
#include <signal.h>
#include <sys/param.h>
#include "owl.h"

int main(int argc, char **argv, char **env) {
  WINDOW *recwin, *sepwin, *typwin, *msgwin;
  owl_editwin *tw;
  owl_popwin *pw;
  int j, ret, initialsubs, debug, newzephyrs;
  struct sigaction sigact;
  char *configfile, *tty;
  char buff[LINE], startupmsg[LINE];

  configfile=NULL;
  tty=NULL;
  debug=0;
  if (argc>0) {
    argv++;
    argc--;
  }
  while (argc>0) {
    if (!strcmp(argv[0], "-n")) {
      initialsubs=0;
      argv++;
      argc--;
    } else if (!strcmp(argv[0], "-c")) {
      if (argc<2) {
	fprintf(stderr, "Too few arguments to -c\n");
	usage();
	exit(1);
      }
      configfile=argv[1];
      argv+=2;
      argc-=2;
    } else if (!strcmp(argv[0], "-t")) {
      if (argc<2) {
	fprintf(stderr, "Too few arguments to -t\n");
	usage();
	exit(1);
      }
      tty=argv[1];
      argv+=2;
      argc-=2;
    } else if (!strcmp(argv[0], "-d")) {
      debug=1;
      argv++;
      argc--;
    } else if (!strcmp(argv[0], "-v")) {
      printf("This is owl version %s\n", OWL_VERSION_STRING);
      exit(0);
    } else {
      fprintf(stderr, "Uknown argument\n");
      usage();	      
      exit(1);
    }
  }

  /* zephyr init */
  if ((ret = ZInitialize()) != ZERR_NONE) {
    com_err("owl",ret,"while initializing");
    exit(1);
  }
  if ((ret = ZOpenPort(NULL)) != ZERR_NONE) {
    com_err("owl",ret,"while opening port");
    exit(1);
  }

  /* signal handler */
  sigact.sa_handler=sig_handler;
  sigemptyset(&sigact.sa_mask);
  sigact.sa_flags=0;
  sigaction(SIGWINCH, &sigact, NULL);

  /* screen init */
  sprintf(buff, "TERMINFO=%s", TERMINFO);
  putenv(buff);
  initscr();
  raw();
  noecho();
  intrflush(stdscr,FALSE);
  keypad(stdscr,TRUE);
  nodelay(stdscr,1);
  meta(stdscr,TRUE);
  clear();
  refresh();

  /* owl init */
  owl_global_init(&g);
  if (debug) owl_global_set_debug(&g);
  owl_global_set_tty(&g, tty);
  owl_global_set_mode(&g, OWL_MODE_READCONFIG);
  owl_readconfig(configfile);
  owl_config_execute("owl::startup();");
  owl_function_debugmsg("-- Owl Startup --");
  
  /* hold on to the window names for convenience */
  msgwin=owl_global_get_curs_msgwin(&g);
  recwin=owl_global_get_curs_recwin(&g);
  sepwin=owl_global_get_curs_sepwin(&g);
  typwin=owl_global_get_curs_typwin(&g);

  tw=owl_global_get_typwin(&g);

  sepbar(NULL);
  wrefresh(sepwin);

  /* welcome message */
  sprintf(buff, "Welcome to owl version %s", OWL_VERSION_STRING);
  strcpy(startupmsg, "*********************************************************************\n");
  strcat(startupmsg, buff);
  strcat(startupmsg, "\n\n");
  strcat(startupmsg, "This is a beta release.  Please add yourself to owl-users@mit.edu for\n");
  strcat(startupmsg, "information on future releases\n");
  strcat(startupmsg, "*********************************************************************\n");
  owl_function_adminmsg("", startupmsg);
  owl_function_makemsg(buff);

  /* load zephyr subs */
  ret=loadsubs(NULL);
  if (ret!=-1) {
    owl_global_add_userclue(&g, OWL_USERCLUE_CLASSES);
  }

  /* load login subs */
  if (owl_global_get_loginsubs(&g)) {
    loadloginsubs(NULL);
  }

  /* zlog in if we need to */
  if (owl_global_get_startuplogin(&g)) {
    owl_function_zlog_in();
  }

  owl_global_set_mode(&g, OWL_MODE_INTERACTIVE);
  while (1) {

    /* if a resize has been scheduled, deal with it */
    owl_global_resize(&g);

    /* these are here in case a resize changes the windows */
    msgwin=owl_global_get_curs_msgwin(&g);
    recwin=owl_global_get_curs_recwin(&g);
    sepwin=owl_global_get_curs_sepwin(&g);
    typwin=owl_global_get_curs_typwin(&g);

    /* grab incoming zephyrs */
    newzephyrs=0;
    while(ZPending()) {
      ZNotice_t notice;
      struct sockaddr_in from;
      owl_message *m;

      newzephyrs=1;

      ZReceiveNotice(&notice, &from);

      /* is this an ack from a zephyr we sent? */
      if (owl_zephyr_notice_is_ack(&notice)) {
	owl_zephyr_handle_ack(&notice);
	continue;
      }
      
      /* if it's a ping we're not viewing pings then skip it */
      if (!owl_global_is_rxping(&g) && !strcasecmp(notice.z_opcode, "ping")) {
	continue;
      }

      /* create the new message */
      m=malloc(sizeof(owl_message));
      owl_message_create_from_zephyr(m, &notice);
      owl_messagelist_append_element(owl_global_get_msglist(&g), m);

      /* do we need to autoreply? */
      if (owl_global_is_zaway(&g)) {
	owl_zephyr_zaway(m);
      }

      /* ring the bell if it's a personal */
      if (owl_global_is_personalbell(&g) && owl_message_is_personal(m)) {
	owl_function_beep();
	owl_global_set_needrefresh(&g);
      }

      /* log it if we need to */
      if (owl_global_is_logging(&g) || owl_global_is_classlogging(&g)) {
	owl_log_incoming(m);
      }
    }

    /* redisplay */
    if (newzephyrs) {
      sepbar(NULL);
      owl_mainwin_redisplay(owl_global_get_mainwin(&g));
      if (owl_popwin_is_active(owl_global_get_popwin(&g))) {
	owl_popwin_refresh(owl_global_get_popwin(&g));
      }
      owl_global_set_needrefresh(&g);
    }

    /* if a popwin just came up, refresh it */
    pw=owl_global_get_popwin(&g);
    if (owl_popwin_is_active(pw) && owl_popwin_needs_first_refresh(pw)) {
      owl_popwin_refresh(pw);
      owl_popwin_no_needs_first_refresh(pw);
      owl_global_set_needrefresh(&g);
    }

    /* update the terminal if we need to */
    if (owl_global_is_needrefresh(&g)) {
      /* leave the cursor in the appropriate window */
      if (owl_global_is_typwin_active(&g)) {
	owl_function_set_cursor(typwin);
      } else {
	owl_function_set_cursor(sepwin);
      }
      doupdate();
      owl_global_set_noneedrefresh(&g);
    }

    /* if the popwin is up typed characters go to it */
    if (owl_popwin_is_active(owl_global_get_popwin(&g))) {
      j=owl_popwin_getch(owl_global_get_popwin(&g));
      if (j==ERR) {
	usleep(10);
      }
      continue;
    }

    /* otherwise characters go to recwin or typwin */
    j=wgetch(typwin);
    if (j==ERR) {
      usleep(10);
      continue;
    }
    if (j==27) {
      j=wgetch(typwin);
      if (j==ERR) continue;
      owl_process_key(META(j));
    } else {
      owl_process_key(j);
    }

  }

}

void sig_handler(int sig) {
  if (sig==SIGWINCH) {
    /* we can't inturrupt a malloc here, so it just sets a flag */
    owl_function_resize();
  }
}

void usage() {
  fprintf(stderr, "Usage: owl [-c <configfile>] [-t <ttyname>]\n");
}
