/*
 * zlogger: the monolithic zephyr logger
 *
 * Written by James Kretchmar
 *
 * Copyright 1999,2004 Massachusetts Institute of Technology
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby
 * granted, provided that the above copyright notice and this
 * permission notice appear in all copies and in supporting
 * documentation.  No representation is made about the suitability of
 * this software for any purpose.  It is provided "as is" without
 * express or implied warranty.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/param.h>
#include <syslog.h>
#include <signal.h>
#include <krb5.h>
#include <zephyr/zephyr.h>
#include <arpa/inet.h>
#include <sys/socket.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <ctype.h>
#include <errno.h>
#include <com_err.h>
#include <string.h>
#include <netinet/in.h>
#include "zlogger.h"

/* for some reason this isn't defined */
#ifdef ZLOGGER_HAVE_AFS
extern void setpag(void);
#endif

zlogger_global g;

int main(int argc, char **argv)
{
  char *configfile = NULL, *foo;
  int c, i, ret, noroll;
  extern char *optarg;
  extern int optind;
  int errflg=0, nofork=0, debug=0, help=0;
  time_t tixtime, substime;

  zlogger_global_init(&g);

  while ((c = getopt(argc, argv, "fdhc:")) != EOF)
    switch (c) {
    case 'c':
      configfile=optarg;
      break;
    case 'f':
      nofork++;
      break;
    case 'd':
      debug++;
      break;
    case '?':
      errflg++;
    case 'h':
      help++;
    }
  
  if (help || errflg || (configfile == NULL)) {
    fprintf(stderr, "zlogger version %s\n", ZLOGGER_VERSION_STRING);
    fprintf(stderr, "Usage: %s [-h] [-f] [-d] -c <filename>\n", argv[0]);
    fprintf(stderr, "        -h    print help message\n");
    fprintf(stderr, "        -f    don't fork\n");
    fprintf(stderr, "        -d    print debugging (and don't fork)\n");
    fprintf(stderr, "        -c    specify configuration file\n");
    exit(1);
  }

  if (debug) {
    nofork=1;
    zlogger_global_set_debug(&g);
  }

  ret=zlogger_readconfig(configfile);
  if (ret) {
    fprintf(stderr, "Could not open configfile for reading\n");
    exit(1);
  }

  foo=malloc(strlen(zlogger_global_get_v4_ticketfile(&g))+30);
  sprintf(foo, "KRBTKFILE=%s", zlogger_global_get_v4_ticketfile(&g));
  putenv(foo);

  foo=malloc(strlen(zlogger_global_get_v5_ticketfile(&g))+30);
  sprintf(foo, "KRB5CCNAME=%s", zlogger_global_get_v5_ticketfile(&g));
  putenv(foo);

  if (zlogger_global_is_debug(&g)) zlogger_print_conf_debug();

#ifdef ZLOGGER_HAVE_AFS
  if (!zlogger_global_is_noafs(&g)) setpag();
#endif

  /* Do it once interactively */
  ret=zlogger_get_tickets_and_tokens(1);
  if (ret) {
    fprintf(stderr, "Error getting tickets or tokens\n");
    exit(2);
  }

  if (!nofork) {
    for(i=0; i<getdtablesize(); i++) {
      close(i);
    }

    ret=fork();
    if (ret == -1) {
      fprintf(stderr, "couldn't fork, going to die\n");
      exit(2);
    } else if (ret != 0) {
      /* parent exits */
      exit(0);
    }

    setsid();
    chdir("/");
    umask(0);
  }

  openlog("zlogger", LOG_PID, LOG_LOCAL0);

  /*
  ret=zlogger_get_tickets_and_tokens(0);
  if (ret) syslog(LOG_ERR, "Error getting tickets or tokens");
  */

  if ((ret = ZInitialize()) != ZERR_NONE) {
    com_err(argv[0],ret,"while initializing");
    exit(1);
  }
  if ((ret = ZOpenPort(NULL)) != ZERR_NONE) {
    com_err(argv[0],ret,"while opening port");
    exit(1);
  }

  zlogger_zephyr_getsubs();

  zlogger_log_initlogs();
  
  tixtime = substime = time(NULL);
  noroll = 0;
  if (zlogger_global_is_debug(&g)) printf("Entering main loop.\n");
  while (1) {
    time_t now;
    struct tm *nowdate;
    
    sleep(2);
    now=time(NULL);

    if (now-tixtime > (g.newtickets * 60 * 60)) {
      if (zlogger_global_is_debug(&g)) printf("Time for new tickets/tokens.\n");
      zlogger_get_tickets_and_tokens(0);
      tixtime=time(NULL);
    }

    if (now-substime > (g.newsubs * 60)) {
      if (zlogger_global_is_debug(&g)) printf("Time for new subs.\n");
      zlogger_zephyr_getsubs();
      substime=time(NULL);
    }

    nowdate=localtime(&now);
    if (!noroll && (nowdate->tm_hour == 3)) {
      if (zlogger_global_is_debug(&g)) printf("Time to roll logs.\n");
      noroll = 1;
      zlogger_log_turnover(&g);
    } else if (nowdate->tm_hour == 4) {
      noroll = 0;
    }

    while (ZPending()) {
      static ZNotice_t notice;
      struct sockaddr_in from;
      if (zlogger_global_is_debug(&g)) printf("Processing new message.\n");
      ZReceiveNotice(&notice, &from);
      zlogger_log_notice(&notice);
      ZFreeNotice(&notice);
    }
    
  }

  return(0);
}

int zlogger_readconfig(char *filename)
{
  FILE *f;
  char line[LINE];
  int linecounter = 0;
  int i, k, ret;
  char **arg, *foo;
  zlogger_entry *e;
  zlogger_list *entrylist;

  f = fopen(filename, "r");
  if (!f) {
    perror(filename);
    return(-1);
  }

  entrylist=zlogger_global_get_entrylist(&g);
  e=NULL;

  while(fgets(line, sizeof(line), f)!=NULL) {
    linecounter++;

    arg=(char **)atokenize(line, " \t\n", &i);
    if (i==0) continue;
    if (arg[0][0] == '#') continue;

    i--; /* so i is the number of args _after_ the keyword arg[0] */

    if (!strcasecmp(arg[0], "ticketfile")) {
      if (i!=1) argserr(arg[0], linecounter);
      zlogger_global_set_ticketfile(&g, arg[1]);

    } else if (!strcasecmp(arg[0], "keytab")) {
      if (i!=1) argserr(arg[0], linecounter);
      zlogger_global_set_keytab(&g, arg[1]);

    } else if (!strcasecmp(arg[0], "kservice")) {
      if (i!=1) argserr(arg[0], linecounter);
      zlogger_global_set_kservice(&g, arg[1]);

    } else if (!strcasecmp(arg[0], "newtickets")) {
      if (i!=1) argserr(arg[0], linecounter);
      zlogger_global_set_newtix_hours(&g, atoi(arg[1]));

    } else if (!strcasecmp(arg[0], "newsubs")) {
      if (i!=1) argserr(arg[0], linecounter);
      zlogger_global_set_newsubs_minutes(&g, atoi(arg[1]));

    } else if (!strcasecmp(arg[0], "aklog")) {
      if (i!=1) argserr(arg[0], linecounter);
      zlogger_global_set_aklog(&g, arg[1]);

    } else if (!strcasecmp(arg[0], "noafs")) {
      if (i!=0) argserr(arg[0], linecounter);
      zlogger_global_set_noafs(&g);

    } else if (!strcasecmp(arg[0], "cells")) {
      if (i<1) argserr(arg[0], linecounter);
      foo=malloc(50*i);
      strcpy(foo, "");
      for (k=0; k<i; k++) {
	strcat(foo, arg[k+1]);
	strcat(foo, " ");
      }
      zlogger_global_set_cells(&g, foo);
      free(foo);
    } else if (!strcasecmp(arg[0], "define")) {
      /* create a new entry and put it on the list */
      e=malloc(sizeof(zlogger_entry));
      if (i==0) {
	zlogger_entry_init(e, "<unnamed>");
      } else if (i==1) {
	zlogger_entry_init(e, arg[1]);
      } else {
	argserr(arg[0], linecounter);
      }
      zlogger_list_append_element(entrylist, e);
    } else if (!strcasecmp(arg[0], "class")) {
      if (i!=1) argserr(arg[0], linecounter);
      if (!e) keyerr(arg[0], linecounter);
      zlogger_entry_set_class(e, arg[1]);
    } else if (!strcasecmp(arg[0], "instance")) {
      if (i!=1) argserr(arg[0], linecounter);
      if (!e) keyerr(arg[0], linecounter);
      ret=zlogger_entry_set_instance(e, arg[1]);
      if (ret) regexerr(arg[1], linecounter);
    } else if (!strcasecmp(arg[0], "opcode")) {
      if (i!=1) argserr(arg[0], linecounter);
      if (!e) keyerr(arg[0], linecounter);
      ret=zlogger_entry_set_opcode(e, arg[1]);
      if (ret) regexerr(arg[1], linecounter);
    } else if (!strcasecmp(arg[0], "logfile")) {
      if (i!=1) argserr(arg[0], linecounter);
      if (!e) keyerr(arg[0], linecounter);
      zlogger_entry_set_logfile(e, arg[1]);
      if (!strstr(arg[1], "%i")) zlogger_entry_set_noparse(e);
    } else if (!strcasecmp(arg[0], "nolog")) {
      if (i!=1) argserr(arg[0], linecounter);
      if (!e) keyerr(arg[0], linecounter);
      if (!strcasecmp(arg[1], "yes")) {
	zlogger_entry_set_nolog_on(e);
      } else {
	zlogger_entry_set_nolog_off(e);
      }
    } else if (!strcasecmp(arg[0], "logroll")) {
      if (i!=2) argserr(arg[0], linecounter);
      if (!e) keyerr(arg[0], linecounter);
      if (!strcasecmp(arg[1], "yes")) {
	zlogger_entry_set_logroll_on(e);
	zlogger_entry_set_loglife_days(e, atoi(arg[2]));
      }
    } else {
      fprintf(stderr, "Unknown keyword '%s' in configfile at line %i\n", arg[0], linecounter);
      exit(1);
    }

    for (k=0; k<i+1; k++) {
      if (arg[k]) free (arg[k]);
    }
    free(arg);
  }
  fclose(f);

  return(0);
}

void argserr(char *key, int line)
{
  fprintf(stderr, "Wrong number of arguments for '%s' on line %i in configfile\n", key, line);
  exit(1);
}

void keyerr(char *key, int line)
{
  fprintf(stderr, "Found keyword '%s' before a 'define' on line %i in the config file\n", key, line);
  exit(1);
}

void regexerr(char *reg, int line)
{
  fprintf(stderr, "Invalid regular expression %s on line %i in config file\n", reg, line);
  exit(1);
}

int zlogger_get_tickets_and_tokens(int interactive)
{
  char *buff;
  int ret;

  ret=k54srvutil(zlogger_global_get_kservice(&g),
		 zlogger_global_get_keytab(&g),
		 interactive);

#ifdef ZLOGGER_HAVE_AFS
  if (!zlogger_global_is_noafs(&g)) {
    buff=malloc(strlen(zlogger_global_get_aklog(&g)) +
		strlen(zlogger_global_get_cells(&g)) +
		100);
    sprintf(buff, "%s %s", zlogger_global_get_aklog(&g), zlogger_global_get_cells(&g));
    system(buff);
    free(buff);
  }
#endif
  return(ret);
}

void zlogger_zephyr_getsubs()
{
  zlogger_entry *e;
  zlogger_list *entrylist;
  ZSubscription_t sub;
  int ret, i, j;

  entrylist=zlogger_global_get_entrylist(&g);
  j=zlogger_list_get_size(entrylist);
  for (i=0; i<j; i++) {
    e=zlogger_list_get_element(entrylist, i);
    sub.zsub_class = zlogger_entry_get_class(e);
    sub.zsub_classinst = "*";
    sub.zsub_recipient = "*";
    if ((ret = ZSubscribeToSansDefaults(&sub,1,NULL)) != ZERR_NONE) {
      syslog(LOG_CRIT, "error %m getting subs");
    }
  }
}

void zlogger_log_notice(ZNotice_t *notice)
{
  zlogger_entry *e;
  zlogger_list *entrylist;
  int i, j;

  entrylist=zlogger_global_get_entrylist(&g);

  /* ingore pings */
  if (!strcasecmp(notice->z_opcode, "ping")) return;

  /* gross, I know */
  j=zlogger_list_get_size(entrylist);
  for (i=0; i<j; i++) {
    e=zlogger_list_get_element(entrylist, i);

    /* does the notice match this entry? */
    if (!zlogger_entry_is_match(e, notice->z_class, notice->z_class_inst, notice->z_opcode)) continue;
      
    /* now check for nolog */
    if (zlogger_entry_is_nolog(e)) {
      if (!strcasecmp(notice->z_opcode, "nolog")) continue;
      if (!strcasecmp(notice->z_class_inst, "nolog")) continue;
    }

    if (zlogger_global_is_debug(&g)) printf("Message on class %s matchs entry %s\n", notice->z_class, zlogger_entry_get_name(e));

    /* if we made it this far go ahead and log it */
    zlogger_log_writelog(e, notice);
  }
}

void zlogger_log_writelog(zlogger_entry *e, ZNotice_t *notice)
{
  char *time;
  struct hostent *hent;
  FILE *f;
  char *ptr, *message;
  int ret;

  message=malloc(notice->z_message_len+20);
  memcpy(message, notice->z_message, notice->z_message_len);
  memset(message + notice->z_message_len, 0,1);
  
  ptr=memchr(message, '\0', notice->z_message_len);
  if (!ptr) ptr=message;
  else ptr++;
  
  time=ctime((time_t *) &notice->z_time.tv_sec);
  memset(time+strlen(time)-1, 0, 1);
  
  hent=gethostbyaddr((void *) &(notice->z_sender_addr), sizeof(notice->z_sender_addr), AF_INET);

  downstr(notice->z_class);
  downstr(notice->z_class_inst);
  downstr(notice->z_opcode);

  if (!strcmp(notice->z_sender+(strlen(notice->z_sender)-strlen("@ATHENA.MIT.EDU")), "@ATHENA.MIT.EDU"))
    memset(notice->z_sender+(strlen(notice->z_sender)-strlen("@ATHENA.MIT.EDU")), 0, 1);

  
  if (zlogger_entry_is_noparse(e)) {
    /* if logfilenoparse==1 then we're class style logging */
    securefile(zlogger_entry_get_logfile(e));
    f=fopen(zlogger_entry_get_logfile(e), "a");
    if (!f) {
      syslog(LOG_ERR, "cannot open file for writing: %m");
    } else {
      ret=fprintf(f, "Instance: %s Time: %s", notice->z_class_inst, time);
      if (ret<0) syslog(LOG_ERR, "error writing to file: %m");
      ret=fprintf(f, " Host: %s\n", (hent && hent->h_name) ? hent->h_name : inet_ntoa(notice->z_sender_addr));
      if (ret<0) syslog(LOG_ERR, "error writing to file: %m");
      ret=fprintf(f, "From: %s <%s>\n", notice->z_message, notice->z_sender);
      if (ret<0) syslog(LOG_ERR, "error writing to file: %m");
      ret=fprintf(f, "\n%s\n",ptr);
      if (ret<0) syslog(LOG_ERR, "error writing to file: %m");
      ret=fclose(f);
      if (ret) syslog(LOG_ERR, "error closing file: %m");
    }
  } else {
    /* it's instance style logging, figure out the filename */
    char *replacement, *newname;
      
    if (is_weird(notice->z_class_inst)) {
      replacement=strdup("weird");
    } else {
      /* is_weird checks the length */
      replacement=strdup(notice->z_class_inst);
    }

    newname=zlogger_util_replace(zlogger_entry_get_logfile(e), "%i", replacement);
    securefile(newname);
    f=fopen(newname, "a");
    if (!f) {
      syslog(LOG_ERR, "cannot open file for writing: %m");
    } else {
      ret=fprintf(f, "Auth: %s ", notice->z_auth ? "yes" : "no");
      if (ret<0) syslog(LOG_ERR, "error writing to file: %m");
      ret=fprintf(f, " Time: %s", time);
      if (ret<0) syslog(LOG_ERR, "error writing to file: %m");
      ret=fprintf(f, " Host: %s\n", (hent && hent->h_name) ? hent->h_name : inet_ntoa(notice->z_sender_addr));
      if (ret<0) syslog(LOG_ERR, "error writing to file: %m");
      ret=fprintf(f, "From: %s <%s>\n", notice->z_message, notice->z_sender);
      if (ret<0) syslog(LOG_ERR, "error writing to file: %m");
      if (is_weird(notice->z_class_inst)) ret=fprintf(f, "Real Instance: %s\n", notice->z_class_inst);
      if (ret<0) syslog(LOG_ERR, "error writing to file: %m");
      ret=fprintf(f, "\n%s\n",ptr);
      if (ret<0) syslog(LOG_ERR, "error writing to file: %m");
      ret=fclose(f);
      if (ret) syslog(LOG_ERR, "error closing file: %m");
    }
    free(newname);
  }
  free(message);
}

void zlogger_log_turnover()
{
  char old[1024], new[1024];
  zlogger_entry *e;
  zlogger_list *entrylist;
  int i, fd, x, y;
  char *logfile;

  entrylist=zlogger_global_get_entrylist(&g);
  y=zlogger_list_get_size(entrylist);
  for (x=0; x<y; x++) {
    e=zlogger_list_get_element(entrylist, x);
    if (!zlogger_entry_is_logroll(e)) continue;

    /* This is really a bug, should be fixed */
    if (!zlogger_entry_is_noparse(e)) continue;

    logfile=zlogger_entry_get_logfile(e);
    for (i=zlogger_entry_get_loglife_days(e)-1; i>=0; i--) {
      sprintf(old, "%s.%i", logfile, i);
      sprintf(new, "%s.%i", logfile, i+1);
      rename(old, new); /* ignoring errors for now */
    }
    securefile(logfile);
    sprintf(new, "%s.0", logfile);
    rename(logfile, new);
    fd=open(logfile, O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
    close(fd);
  }
}

void zlogger_log_initlogs()
{
  zlogger_entry *e;
  zlogger_list *entrylist;
  int fd, i, j;

  entrylist=zlogger_global_get_entrylist(&g);
  j=zlogger_list_get_size(entrylist);
  for (i=0; i<j; i++) {
    e=zlogger_list_get_element(entrylist, i);

    if (!zlogger_entry_is_logroll(e)) continue;

    /* same bug as above */
    if (!zlogger_entry_is_noparse(e)) continue;

    fd=open(zlogger_entry_get_logfile(e), O_WRONLY | O_CREAT, S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH);
    close(fd);
  }
}

void zlogger_print_conf_debug()
{
  int i, j;
  zlogger_list *entrylist;
  zlogger_entry *e;
  
  printf("DEBUG: setting are:\n");
  printf("  debugging: %s\n", zlogger_global_is_debug(&g)?"on":"off");
  printf("  v4tkfile : %s\n", zlogger_global_get_v4_ticketfile(&g));
  printf("  v5tkfile : %s\n", zlogger_global_get_v5_ticketfile(&g));
  printf("  keytab   : %s\n", zlogger_global_get_keytab(&g));
  printf("  kservice : %s\n", zlogger_global_get_kservice(&g)?zlogger_global_get_kservice(&g):"NULL");
  printf("  newtix   : %i hr\n", zlogger_global_get_newtix_hours(&g));
  printf("  newsubs  : %i min\n", zlogger_global_get_newsubs_minutes(&g));
  printf("  aklog    : %s\n", zlogger_global_get_aklog(&g));
  printf("  cells    : %s\n", zlogger_global_get_cells(&g));
  printf("  noafs    : %i\n", zlogger_global_is_noafs(&g));
  printf("DEBUG: envrionment is\n");
  printf("  KRBTKFILE  : %s\n", getenv("KRBTKFILE")?getenv("KRBTKFILE"):"");
  printf("  KRB5CCANME : %s\n", getenv("KRB5CCNAME")?getenv("KRB5CCNAME"):"");
	 
  printf("DEBUG: entries are:\n");
  entrylist=zlogger_global_get_entrylist(&g);
  j=zlogger_list_get_size(entrylist);
  for (i=0; i<j; i++) {
    e=zlogger_list_get_element(entrylist, i);

    printf("  Entry Name: %s\n", zlogger_entry_get_name(e));
    printf("    Class: %s\n", zlogger_entry_get_class(e));
    printf("    Instance: xxx\n");
    printf("    Opcode: xxx\n");
    printf("    NoParse: %i\n", zlogger_entry_is_noparse(e));
    printf("    Logfile: %s\n", zlogger_entry_get_logfile(e));
    printf("    NoLog: %i\n", zlogger_entry_is_nolog(e));
    printf("    LogRoll: %i\n", zlogger_entry_is_logroll(e));
    printf("    LogLife: %i\n", zlogger_entry_get_loglife_days(e));
  }
}
